package org.gcube.resourcemanagement.resource;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonAnyGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonAnySetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.InternalServerErrorException;
import jakarta.ws.rs.WebApplicationException;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public abstract class Instance implements Comparable<Instance> {

     private static Logger logger = LoggerFactory.getLogger(Instance.class);

    protected static Set<String> excludedFacetKeys;
    
    static {
        excludedFacetKeys = new HashSet<>();
        excludedFacetKeys.add(Facet.ID_PROPERTY);
        excludedFacetKeys.add(Facet.METADATA_PROPERTY);
        excludedFacetKeys.add(Facet.TYPE_PROPERTY);
        excludedFacetKeys.add(Facet.SUPERTYPES_PROPERTY);
        excludedFacetKeys.add(Facet.EXPECTED_TYPE_PROPERTY);
        excludedFacetKeys.add(Facet.CONTEXTS_PROPERTY);
    }

    public static <I extends Instance> Set<I> getAsInstances(Class<I> iClass, Collection<Resource> resources) throws WebApplicationException {
        Set<I> instances = new HashSet<>();
        for(Resource r : resources){
            I i = null;
            try {
                i = iClass.getConstructor().newInstance();
                i.setResource(r);
            } catch (Exception e) {
                StringBuffer error = new StringBuffer();
                error.append("Error while creating a new instance of ");
                error.append(iClass.getName());
                logger.error(error.toString(), e);
                throw new InternalServerErrorException(error.toString(),e);
            }
            instances.add(i);
        }
        return instances;
    }

    public static <I extends Instance> Set<I> getAsInstances(Class<I> iClass, Resource... resources) throws WebApplicationException {
        Set<I> instances = new HashSet<>();
        for(Resource r : resources){
            I i = null;
            try {
                i = iClass.getConstructor(r.getClass()).newInstance(r);
            } catch (Exception e) {
                StringBuffer error = new StringBuffer();
                error.append("Error while creating a new instance of ");
                error.append(iClass.getName());
                logger.error(error.toString(), e);
                throw new InternalServerErrorException(error.toString(),e);
            }
            instances.add(i);
        }
        return instances;
    }

    @JsonIgnore
    protected ObjectMapper mapper;

    public Instance() {
        this.mapper = new ObjectMapper();
        this.mapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
    }
    
    protected abstract void setIdentifingProperties(Resource r) throws WebApplicationException;

    public void setResource(Resource r) throws WebApplicationException {
        setID(r.getID());
        setType(r.getTypeName());
    }

    /**
     * UUID of the instance
     */
    protected UUID id;
    
    /**
     * The type of the instance
     */
    protected String type;

    /**
     * Properties which describe the instance.
     * For example for any GCubeResource all the properties of the Facet which identify the instance.
     * 
     * e.g. EService -- isIdentifiedBy --> SoftwareFacet
     * 
     * All the properties (mandatory or not) defined in the SoftwareFacet instance identifiyng the EService
     * 
     */
    protected Map<String,JsonNode> properties;

    protected Set<Instance> mandatoryRelatedResources;

    protected Set<DerivatedRelatedResourceGroup<Instance>> derivatedRelatedResources;

    protected List<IdentifiableElement> newInstances;

    /**
     * The elements that have been added to the context.
     * It is used to keep track of the elements that have been added to the context
     * to inform the clinet of the result of add to context.
     * It is not used when we read an instace ofr add to context operation.
     */
    protected List<ERElement> addedElements;

    @JsonGetter(value = "id")
    public UUID getID() {
        return id;
    }
    
    @JsonSetter(value = "id")
    public void setID(UUID id) {
        this.id = id;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }
    
    @JsonAnyGetter
    public Map<String, JsonNode> getProperties() {
        return properties;
    }
    
    public void setProperties(Map<String, JsonNode> properties) {
        this.properties = properties;
    }
    
    @JsonAnySetter
    public void addProperty(String key, JsonNode value){
        if(this.properties == null){
            this.properties = new java.util.HashMap<>();
        }
        this.properties.put(key, value);
    }

    @JsonGetter
    public Set<Instance> getMandatoryRelatedResources() {
        return mandatoryRelatedResources;
    }

    public void setMandatoryRelatedResources(Set<Instance> mandatoryRelatedResources) {
        this.mandatoryRelatedResources = mandatoryRelatedResources;
    }

    @JsonSetter
    public void addMandatoryRelatedResource(Instance instance){
        if(this.mandatoryRelatedResources == null){
            this.mandatoryRelatedResources = new HashSet<>();
        }
        this.mandatoryRelatedResources.add(instance);
    }

    @JsonIgnore
    public void addMandatoryRelatedResources(Set<Instance> instances){
        if(this.mandatoryRelatedResources == null){
            this.mandatoryRelatedResources = new HashSet<>();
        }
        this.mandatoryRelatedResources.addAll(instances);
    }

    @JsonGetter
    public Set<DerivatedRelatedResourceGroup<Instance>> getDerivatedRelatedResources() {
        return derivatedRelatedResources;
    }

    public void setDerivatedRelatedResources(Set<DerivatedRelatedResourceGroup<Instance>> derivatedRelatedResources) {
        this.derivatedRelatedResources = derivatedRelatedResources;
    }

    @JsonSetter
    public void addDerivatedRelatedResource(DerivatedRelatedResourceGroup<Instance> derivatedRelatedResource){
        if(this.derivatedRelatedResources == null){
            this.derivatedRelatedResources = new HashSet<>();
        }
        this.derivatedRelatedResources.add(derivatedRelatedResource);
    }
    
    @JsonIgnore
    public void addDerivatedRelatedResources(Set<DerivatedRelatedResourceGroup<Instance>> derivatedRelatedResources){
        if(this.derivatedRelatedResources == null){
            this.derivatedRelatedResources = new HashSet<>();
        }
        this.derivatedRelatedResources.addAll(derivatedRelatedResources);
    }

    @JsonGetter
    public List<IdentifiableElement> getNewInstances() {
        return newInstances;
    }

    public void setNewInstances(List<IdentifiableElement> newInstances) {
        this.newInstances = newInstances;
    }

    @JsonSetter
    public void addNewInstance(IdentifiableElement newInstance){
        if(this.newInstances == null){
            this.newInstances = new ArrayList<>();
        }
        this.newInstances.add(newInstance);
    }

    @JsonIgnore
    public void addNewInstances(List<IdentifiableElement> newInstances){
        if(this.newInstances == null){
            this.newInstances = new ArrayList<>();
        }
        if(newInstances!=null && !newInstances.isEmpty()){
            this.newInstances.addAll(newInstances);
        }
    }

    @Override
    public String toString(){
        try {
            return mapper.writeValueAsString(this);
        } catch (Exception e) {
            logger.error("Error while serializing the instance", e);
            return super.toString();
        }
    }

    /**
     * All the ERElement added as result of addToContext
     * The added element are got from the return of the addToContext operation
     * from the resource registry.
     * @return
     */
    public List<ERElement> getAddedElements() {
        return addedElements;
    }

    public void setAddedElements(List<ERElement> addedElements) {
        this.addedElements = addedElements;
    }

    public void addAddedElement(ERElement addedElement){
        if(this.addedElements == null){
            this.addedElements = new ArrayList<>();
        }
        this.addedElements.add(addedElement);
    }

    public void addAddedElements(List<ERElement> addedElements){
        if(this.addedElements == null){
            this.addedElements = new ArrayList<>();
        }
        if(addedElements!=null && !addedElements.isEmpty()){
            this.addedElements.addAll(addedElements);
        }
    }

    public void addAddedElements(ERElement... addedElements){
        if(this.addedElements == null){
            this.addedElements = new ArrayList<>();
        }
        if(addedElements!=null && addedElements.length>0){
            for(ERElement addedElement : addedElements){
                this.addedElements.add(addedElement);
            }
        }
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        result = prime * result + ((type == null) ? 0 : type.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Instance other = (Instance) obj;
        if (id == null) {
            if (other.id != null)
                return false;
        } else if (!id.equals(other.id))
            return false;
        if (type == null) {
            if (other.type != null)
                return false;
        } else if (!type.equals(other.type))
            return false;
        return true;
    }

    @Override
    public int compareTo(Instance o) {
        int idComparison = this.id.compareTo(o.id);
        if (idComparison != 0) {
            return idComparison;
        }
        return this.type.compareTo(o.type);
    }

}
