package org.gcube.resourcemanagement.analyser;

import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.UUID;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.tree.Tree;
import org.gcube.informationsystem.types.knowledge.TypesKnowledge;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.resourcemanagement.resource.Instance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.InternalServerErrorException;
import jakarta.ws.rs.WebApplicationException;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class InstanceAnalyserFactory {

    /**
     * Logger
     */
    protected static Logger logger = LoggerFactory.getLogger(InstanceAnalyserFactory.class);

    protected static Map<String, InstanceAnalyser<Resource, Instance>> instances;
    protected static Map<String, InstanceAnalyser<Resource, Instance>> polymorphicInstances;

    
    protected static void addPolymorphicInstanceAnalyser(String type, InstanceAnalyser<Resource, Instance> analyser){
        if(!analyser.polymorphic()){
            return;
        }
        TypesKnowledge tk = TypesKnowledge.getInstance();
        Tree<Type> resourceTree = tk.getModelKnowledge().getTree(AccessType.RESOURCE);
        Node<Type> node = resourceTree.getNodeByIdentifier(type);
        Set<Node<Type>> descendants = node.getDescendants();
        for (Node<Type> child : descendants) {
            String childTypeName = child.getIdentifier();
            if(instances.containsKey(childTypeName)){
                /*
                 * Not adding the polymorphic analyser if there is a specific analyser for the child type
                 * and removing the the list of polymorphic analysers if any
                 */
                if(polymorphicInstances.containsKey(childTypeName)){
                    polymorphicInstances.remove(childTypeName);
                }
                continue;
            }else{
                if(!polymorphicInstances.containsKey(childTypeName)){
                    polymorphicInstances.put(childTypeName, analyser);
                }else{
                    evaluateBestInstanceAnalyser(childTypeName, analyser, resourceTree);
                }
            }
        }
    }

    protected static void evaluateBestInstanceAnalyser(String type, InstanceAnalyser<Resource, Instance> candidatedAnalyser, Tree<Type> resourceTree) throws WebApplicationException{
        InstanceAnalyser<Resource, Instance> currentAnalyser = polymorphicInstances.get(type);
        String typeOfCurrentAnalyser = currentAnalyser.getType();
        String typeOfCandidatedAnalyser = candidatedAnalyser.getType();
        
        if(typeOfCurrentAnalyser.compareTo(typeOfCandidatedAnalyser) == 0){
            throw new InternalServerErrorException("Two polymorphic analysers with the same type found. This is not allowed");
        } else if(resourceTree.isChildOf(typeOfCurrentAnalyser, typeOfCandidatedAnalyser)){
            /*
            * If the type of the candidated analyser is a child of the type of the current analyser, 
            * the candidated analyser is the best one because it is more specific.
            */
            // Remove the current list of analysers because we found a more specific analyser
            polymorphicInstances.remove(type);
            // Add the candidated analyser because it is more specific
            polymorphicInstances.put(type, candidatedAnalyser);
        }
    }
    
    protected static void analyseInstanceAnalyser(InstanceAnalyser<Resource, Instance> analyser){
        try {
            String name = analyser.getName();
            logger.debug("{} {} found", name, InstanceAnalyser.class.getSimpleName());
            String type = analyser.getType();
            if(instances.get(type)!=null){
                throw new InternalServerErrorException("Two analysers with the same type found. This is not allowed");
            }
            instances.put(type, analyser);
            addPolymorphicInstanceAnalyser(type, analyser);
        } catch (Exception e) {
            logger.error("{} {} not initialized correctly. It will not be used", analyser.getName(), InstanceAnalyser.class.getSimpleName());
        }
    }

    static {
        InstanceAnalyserFactory.instances = new HashMap<>();
        InstanceAnalyserFactory.polymorphicInstances = new HashMap<>();

        @SuppressWarnings("rawtypes")
        ServiceLoader<InstanceAnalyser> serviceLoader = ServiceLoader.load(InstanceAnalyser.class);
        for (@SuppressWarnings("rawtypes") InstanceAnalyser analyser : serviceLoader) {
            @SuppressWarnings("unchecked")
            InstanceAnalyser<Resource, Instance> instanceAnalyser = analyser;
            analyseInstanceAnalyser(instanceAnalyser);
        }

        if(logger.isTraceEnabled()){
            logger.trace("{} initialized with the following {}s:", InstanceAnalyserFactory.class.getSimpleName(), InstanceAnalyserFactory.class.getSimpleName());
            for (String type : instances.keySet()) {
                InstanceAnalyser<Resource, Instance> analyser = instances.get(type);
                logger.trace("{} will be managed managed by {}", type, analyser.getName());
            }

            logger.trace("{} initialized with the following polymorphic {}s:", InstanceAnalyserFactory.class.getSimpleName(), InstanceAnalyserFactory.class.getSimpleName());
            for (String type : polymorphicInstances.keySet()) {
                InstanceAnalyser<Resource, Instance> analyser = polymorphicInstances.get(type);
                logger.trace("{} will be managed managed by {} because is polymorphic", type, analyser.getName());
            }
        }
        
    }

    public static InstanceAnalyser<Resource, Instance> getInstanceAnalyser(String type, UUID id) {
        InstanceAnalyser<Resource, Instance> instanceAnalyser = null;
        if (instances.containsKey(type)) {
            InstanceAnalyser<Resource, Instance> analyser = instances.get(type);
            try {
                @SuppressWarnings("unchecked")
                Class<InstanceAnalyser<Resource, Instance>> clz = (Class<InstanceAnalyser<Resource, Instance>>) analyser.getClass();
                instanceAnalyser = clz.getDeclaredConstructor(String.class, UUID.class).newInstance(type, id);
            } catch (Exception e) {
                logger.error("Error while creating {} instance", InstanceAnalyser.class.getSimpleName(), e);
            }
        }else if(polymorphicInstances.containsKey(type)){
                InstanceAnalyser<Resource, Instance> analyser = polymorphicInstances.get(type);
                try {
                    @SuppressWarnings("unchecked")
                    Class<InstanceAnalyser<Resource, Instance>> clz = (Class<InstanceAnalyser<Resource, Instance>>) analyser.getClass();
                    instanceAnalyser = clz.getDeclaredConstructor(String.class, UUID.class
                    ).newInstance(type, id);
                } catch (Exception e) {
                    logger.error("Error while creating {} instance", InstanceAnalyser.class.getSimpleName(), e);
                }
        } else{
            logger.error("No {} found for type {}", InstanceAnalyser.class.getSimpleName(), type);
        }
        if(instanceAnalyser == null){
            throw new InternalServerErrorException("No instance analyser found for type " + type);

        }
        return instanceAnalyser;
    }

}
