package org.gcube.vremanagement.executor.plugin;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.gcube.common.core.plugins.GCUBEPluginContext;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.types.DescriptiveProperty;
import org.gcube.vremanagement.executor.stubs.TaskDescription;

/**
 * Extends {@link GCUBEPluginContext} to provide the service with information about its plugin.
 * <p>
 * Subclasses <b>must</b> indicate the class of an executable {@link ExecutorTask} (cf. {@link #getTaskClass()}).<br> 
 * They <em>may</em> also provide examples of the any inputs the task may take and any output the task may produce 
 * (cf. {@link #addSampleInput(DescriptiveProperty)}, {@link #addSampleOutput(DescriptiveProperty...)}). <br> 
 * Finally, they <em>may</em> also indicate the time after which task executions that are completed or have failed 
 * can be forgotten (cf. {@link #getTimeToLive()}). 
 * <p>
 * 
 * @author Fabio Simeoni (University of Strathclyde)
 */
public abstract class ExecutorPluginContext extends GCUBEPluginContext {
	
	/**Plugin sample inputs.*/
	private List<DescriptiveProperty> sampleInputs = new ArrayList<DescriptiveProperty>(); 
	
	/**Plugin sample outputs.*/
	private List<DescriptiveProperty> sampleOutputs = new ArrayList<DescriptiveProperty>(); 
	
	/**Default task expiration delay.*/
	public static final int TIME_TO_LIVE = 5;
	
	/**The description of the task.*/
	private TaskDescription description = new TaskDescription();
	
	/**Returns the description of the task. * @return the description.*/
	public TaskDescription getDescription() {return this.description;}
	/**Sets the description of the task.* @param the description.*/
	public void setTaskDescription(TaskDescription description) {this.description = description;}

	/**Return the type of the task.* @return the type.*/
	public abstract Class<? extends ExecutorTask> getTaskClass(); 
	
	/** Returns sample inputs for the task. @return the sample inputs.*/
	public List<DescriptiveProperty> getSampleInputs() {return this.sampleInputs;}

	/**Return sample outputs for the task.* @return the sample outputs.*/
	public List<DescriptiveProperty> getSampleOutputs() {return this.sampleOutputs;}
	
	/**
	 * Adds one or more sample inputs for the task.
	 * <p>
	 * Subclasses typically add sample inputs in {@link #initialise(GCUBEService)}. 
	 * @param inputs the sample inputs. 
	 */
	protected void addSampleInput(DescriptiveProperty inputs) {
		if (inputs!= null)
			this.sampleInputs.addAll(Arrays.asList(inputs));
	}
	
	/**
	 * Adds one ore more sample output for the task.
	 * <p>
	 * Subclasses typically add sample output in {@link #initialise(GCUBEService)}. 
	 * @param outputs the sample outputs. 
	 */
	protected void addSampleOutput(DescriptiveProperty ... outputs) {
		if (outputs!= null)
			this.sampleOutputs.addAll(Arrays.asList(outputs));
	}
	
	
	/**Returns the time-to-live for completed or failed task, in minutes.@return the time-to-live.*/
	public int getTimeToLive(){return TIME_TO_LIVE;}
	
}
