package org.gcube.vremanagement.executor.state;

import java.util.Map;

import org.gcube.common.core.faults.GCUBERetryEquivalentException;
import org.gcube.common.core.plugins.GCUBEPluginManager;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.core.types.DescriptiveProperty;
import org.gcube.common.core.utils.events.GCUBEEvent;
import org.gcube.vremanagement.executor.contexts.ServiceContext;
import org.gcube.vremanagement.executor.contexts.TaskPTContext;
import org.gcube.vremanagement.executor.plugin.ExecutorPluginContext;
import org.gcube.vremanagement.executor.stubs.AnyMap;
import org.gcube.vremanagement.executor.stubs.TaskDescription;
import org.globus.wsrf.ResourceProperty;

/**
 * Specialises {@link GCUBEWSResource} for the singleton resource of the engine. 
 * @author Fabio Simeoni (University of Strathclyde)
 * @author Kostantinos Liaskov (University of Strathclyde)
 */
public class EngineResource extends GCUBEWSResource {

	/** Name of the task descriptions RP.*/
 	static final String RP_TASKS="Task";
 	
	/** RP names. */
	private static String[] RPNames = {RP_TASKS};
	
	/**Extends {@link GCUBEPluginManager.PluginConsumer} to consume plugin management events.*/
	private class PluginConsumer extends GCUBEPluginManager.PluginConsumer<ExecutorPluginContext> {
		/**{@inheritDoc}*/
		@SuppressWarnings("all") public <T1 extends GCUBEPluginManager.PluginTopic, P1 extends ExecutorPluginContext> void onEvent(GCUBEEvent<T1,P1> ... events) {
			logger.trace("received a plugin management event");
			setTaskRP(ServiceContext.getContext().getPluginManager().getPlugins());//update taskRP with currently registered tasks
		};
	}
 	
	/**{@inheritDoc}*/
	protected void initialise(Object... args) throws Exception,IllegalArgumentException {
		logger.trace("subscribing for plugin event management");
		ServiceContext.getContext().getPluginManager().subscribe(new PluginConsumer());
	}
	
	/**
	 * Sets the tasks managed by the engine.
	 * @param context the task.
	 */
	private synchronized void setTaskRP(Map<String,ExecutorPluginContext> tasks) {
		logger.trace("updating taskRP property");
		ResourceProperty taskRP = this.getResourcePropertySet().get(RP_TASKS);
		taskRP.clear();
		for (ExecutorPluginContext context : tasks.values()) {//prepare task description
			TaskDescription description =  context.getDescription();
			description.setName(context.getPlugin().getServiceName());
			description.setDescription(context.getPlugin().getDescription());
			description.setProperty(context.getProperties().values().toArray(new DescriptiveProperty[0]));
			description.setSampleInput(context.getSampleInputs().toArray(new DescriptiveProperty[0]));
			description.setSampleOutput(context.getSampleOutputs().toArray(new DescriptiveProperty[0]));		
			taskRP.add(description);//add description to RP
		}
	}
	
	
	/** 
	 * Returns the context of a plugin with a given name.
	 * @param name the name.
	 * @return the context
	 */
	public ExecutorPluginContext getContext(String name) throws Exception  {
		ExecutorPluginContext taskContext = ServiceContext.getContext().getPluginManager().getPlugins().get(name);
		if (taskContext==null) throw new GCUBERetryEquivalentException("task "+name+" is unknown");
		return taskContext;
	}
	
	/**
	 * Launch a task with a given name and a given input.
	 * @param name the name.
	 * @param input the input.
	 * @return a {@link TaskResource} that models the execution of the task.
	 * @throws Exception if the task did not exist or could not be launched.
	 */
	public TaskResource launch(String name, AnyMap input) throws Exception {
		return (TaskResource) TaskPTContext.getContext().getWSHome().create(getContext(name),input);
	}

	/** {@inheritDoc} */
	public String[] getPropertyNames() {return RPNames;}
}
