package org.gcube.vremanagement.executor.stubs;

import java.util.Arrays;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.globus.wsrf.NotifyCallback;
import org.globus.wsrf.WSNConstants;
import org.oasis.wsn.Subscribe;
import org.oasis.wsn.TopicExpressionType;
import org.oasis.wsn.WSBaseNotificationServiceAddressingLocator;

/**
 * A monitor of tasks in execution.
 * @author Fabio Simeoni (University of Strathclyde)
 */
public abstract class TaskMonitor implements NotifyCallback {

	/** Class logger */
	static GCUBELog logger = new GCUBEClientLog(TaskMonitor.class);
	
	/**Topics relative to task execution.
	 * @author Fabio Simeoni (University of Strathclyde)
	 */
	public static enum TaskTopic{
		STATECHANGE(){
			@Override public QName getQName(){return Utils.TASK_STATE_RP;}
			@Override public void callback(TaskMonitor monitor) throws Exception {monitor.onStateChange(monitor.proxy);}
		},
		OUTPUTCHANGE(){
			@Override public QName getQName(){return Utils.TASK_OUTPUT_RP;}
			@Override public void callback(TaskMonitor monitor) throws Exception {monitor.onOutputChange(monitor.proxy);}
		};
		/**
		 * The topic's qualified name. 
		 * @return the name.
		 */
		abstract QName getQName();
		/**
		 * Invokes the topic's callback on a {@link TaskMonitor}.
		 * @param monitor the monitor.
		 * @throws Exception if the callback failed.
		 */
		abstract void callback(TaskMonitor monitor) throws Exception;
	}
	
		
	/**The monitor's topics.*/
	private List<TaskTopic> topics;
	
	/** Remote task proxy.*/
	private TaskProxy proxy;

	/**
	 * Creates a monitor for one more {@link TaskTopic TaskTopics}.
	 * @param topics (optional) the topics (all by default).
	 */
	public TaskMonitor(TaskTopic ... topics) {
		this.topics = Arrays.asList((topics!=null && topics.length>0)?topics:TaskTopic.values());
	}
	
	
	/**
	 * Returns the monitor's topics.
	 * @return the topics.
	 */
	public List<TaskTopic> getTopics() {return topics;}

	
	/**
	 * Subscribes monitor with remote task.
	 * @throws Exception if the monitor could not be subscribed.
	 */
	protected void subscribe(EndpointReferenceType epr) throws Exception {
		
		for (TaskTopic topic : getTopics()) {
			logger.trace("subscribing for "+topic);
			EndpointReferenceType consumerEPR = MappingRegistry.manager.createNotificationConsumer(this);
			Subscribe subscriptionRequest = new Subscribe();
			subscriptionRequest.setUseNotify(Boolean.TRUE);
			subscriptionRequest.setConsumerReference(consumerEPR);
			TopicExpressionType topicExpression = new TopicExpressionType();
			topicExpression.setDialect(WSNConstants.SIMPLE_TOPIC_DIALECT);
			topicExpression.setValue(topic.getQName());
			subscriptionRequest.setTopicExpression(topicExpression);
			new WSBaseNotificationServiceAddressingLocator().getNotificationProducerPort(epr).subscribe(subscriptionRequest);
		}
		
		try {
			logger.trace("post-subscription delivery");
			proxy = new TaskProxy(epr);
			for (TaskTopic topic : getTopics()) topic.callback(this);
		}
		catch(Exception e) {logger.error("could not process event",e);}
	}
	
	/**{@inheritDoc}*/
	public void deliver(@SuppressWarnings("rawtypes")List topicPath, EndpointReferenceType producer, Object message) {
		try {
			QName topic = (QName) topicPath.get(0); //what's changed?
			logger.trace("processing event for "+topic);
			proxy.synchronize();
			for (TaskTopic t : getTopics()) //if relevant, let topic dispatch to corresponding callback
				if (topic.equals(t.getQName())) t.callback(this);
		}
		catch(Exception e) {logger.error("could not process event",e);}
	}
	
	/**
	 * Callback on state change.
	 * @param proxy a {@link TaskProxy}.
	 */
	protected void onStateChange(TaskProxy proxy) throws Exception {}
	/**
	 * Callback on output change.
	 * @param proxy a {@link TaskProxy}.
	 */
	protected void onOutputChange(TaskProxy proxy) throws Exception {}
}
