package org.gcube.social_networking.server;

import java.rmi.ServerException;
import java.time.Instant;
import java.util.*;

// import jakarta.mail.internet.AddressException;
// import jakarta.mail.internet.InternetAddress;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.*;
import com.datastax.oss.driver.api.querybuilder.QueryBuilder;
import org.apache.commons.lang.NullArgumentException;
import org.gcube.social_networking.socialnetworking.model.shared.*;
import org.gcube.social_networking.socialnetworking.model.shared.exceptions.*;

import org.gcube.social_networking.utils.ParameterNames;
import org.gcube.social_networking.utils.ResourceNames;
import org.gcube.social_networking.utils.Schema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.gcube.social_networking.utils.Schema.*;
import static org.gcube.social_networking.utils.ResourceNames.*;

/**
 * @author Massimiliano Assante ISTI-CNR
 * @author Costantino Perciante ISTI-CNR
 * @author Ahmed Ibrahim ISTI-CNR
 * This class is used for querying and adding data to Cassandra via Datastax High Level API
 */
public final class SocialDBDatastaxDriver implements SocialDBDriver{

    private static final Logger _log = LoggerFactory.getLogger(SocialDBDatastaxDriver.class);

    /**
     * connection instance
     */
    private CassandraClusterConnection conn;

    protected CassandraClusterConnection getConnection() {
        return conn;
    }
    /**
     * use this constructor carefully from test classes
     * @param dropSchema set true if you want do drop the current and set up new one
     */
    protected SocialDBDatastaxDriver(boolean dropSchema) {
        try {
            conn = new CassandraClusterConnection(dropSchema);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
    /**
     * public constructor, no dropping schema is allowed
     */
    public SocialDBDatastaxDriver() {
        try {
            conn = new CassandraClusterConnection(false);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * public constructor, no dropping schema is allowed, infrastructureName is given.
     */
    public SocialDBDatastaxDriver(String infrastructureName) {
        try {
            conn = new CassandraClusterConnection(false, infrastructureName);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    /*
     *
     ********************** 	Generic CRUD	***********************
     *
     */
    @Override
    public Boolean create(IdResource resource) {
        String resourceName = resource.getClass().getSimpleName();
        if(resourceName.equals(ResourceNames.POST)){
            try {
                return savePost((PostWithAttachment) resource);
            } catch (PostTypeNotFoundException | ColumnNameNotFoundException | PostIDNotFoundException |
                     PrivacyLevelTypeNotFoundException e) {
                throw new RuntimeException(e);
            }
        } else if(resourceName.equals(ResourceNames.NOTIFICATION)) {
            return saveNotification((Notification) resource);
        }
        else{
            return null;
        }
    }

    @Override
    public Boolean createChildOf(String parentid, IdResource childResource, String parentResourceName){
        String resourceName = childResource.getClass().getSimpleName();
        if(parentResourceName.equals(ResourceNames.VRE)) {
            if(resourceName.equals(ResourceNames.POST)){
                try {
                    savePostToVRETimeline(parentid, (Post) childResource);
                } catch (PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }

        }else {

            if (resourceName.equals(ResourceNames.COMMENT)) {
                try {
                    return saveComment(parentid, (Comment) childResource);
                } catch (PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            } else if (resourceName.equals(ResourceNames.LIKE)) {
                try {
                    return like(parentid, (Like) childResource);
                } catch (PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return null;

    }

    @Override
    public Boolean BatchCreateChildOfWithParameter(String parentid, String parameterName, String parameterValue, List<IdResource> childResource, String parentResourceName, String childResourceName){
        if(parentResourceName.equals(ResourceNames.VRE)){
            if(childResourceName.equals(ResourceNames.HASHTAG)){
                List<String> hashtags = new ArrayList<>();
                for(IdResource r: childResource){
                    Hashtag h = (Hashtag) r;
                    hashtags.add(h.getHashtag());
                }
                if(parameterName.equals(ResourceNames.POST)){
                    try {
                        return saveHashTags(parameterValue, parentid, hashtags);
                    } catch (PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }else if(parameterName.equals(ResourceNames.COMMENT)){
                    try {
                        return saveHashTagsComment(parameterValue, parentid, hashtags);
                    } catch (CommentIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return null;
    }

    @Override
    public IdResource read(String id, String resourceName) {
        if(resourceName.equals(ResourceNames.POST)){
            try {
                return readPost(id);
            } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | PostIDNotFoundException |
                     ColumnNameNotFoundException e) {
                throw new RuntimeException(e);
            }
        } else if (resourceName.equals(ResourceNames.NOTIFICATION)) {
            try {
                return readNotification(id);
            } catch (NotificationIDNotFoundException | NotificationTypeNotFoundException | ColumnNameNotFoundException e) {
                throw new RuntimeException(e);
            }
        }else if (resourceName.equals(ResourceNames.COMMENT)) {
            try {
                return readComment(id);
            } catch (CommentIDNotFoundException e) {
                throw new RuntimeException(e);
            }

        }else if(resourceName.equals(ResourceNames.INVITE)){
            try {
                return readInvite(id);
            } catch (InviteIDNotFoundException | InviteStatusNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else{
            return null;
        }
    }

    @Override
    public Boolean check(String id, String parameterName, String parameterValue, String resourceName) {
        return null;
    }

    @Override
    public Boolean checkChildOf(String parentid, String parameterName, String parameterValue, String parentResourceName, String childResourceName){
        if(parentResourceName.equals(ResourceNames.USER)){
            if(childResourceName.equals(ResourceNames.NOTIFICATION)){
                if(parameterName.equals(ParameterNames.UNREAD_MESSAGE)){
                    if(parameterValue.equals(String.valueOf(true))){
                        try {
                            return checkUnreadMessagesNotifications(parentid);
                        } catch (NotificationIDNotFoundException | NotificationTypeNotFoundException |
                                 ColumnNameNotFoundException e) {
                            throw new RuntimeException(e);
                        }
                    }
                } else if(parameterName.equals(ParameterNames.UNREAD)){
                    if(parameterValue.equals(String.valueOf(true))){
                        try {
                            return checkUnreadNotifications(parentid);
                        } catch (NotificationTypeNotFoundException | ColumnNameNotFoundException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }

            }
        }
        return null;
    }

    @Override
    public List<IdResource> readAll(String resourceName) {
        if(resourceName.equals(ResourceNames.VRE)){
            return new ArrayList<>(getAllVREIds());
        }
        else{
            return null;
        }
    }
    @Override
    public List<IdResource>readChildOf(String parentid, String parentResourceName, String childResourceName){
        if (parentResourceName.equals(ResourceNames.POST)){
            if(childResourceName.equals(ResourceNames.COMMENT)){
                return getAllChildrenByPost(parentid, COMMENTS);
            } else if (childResourceName.equals(ResourceNames.LIKE)){
                return getAllChildrenByPost(parentid, LIKES);
            } else if (childResourceName.equals(ResourceNames.Attachment)){
                return getAllChildrenByPost(parentid, ATTACHMENTS);
            }
        }else if (parentResourceName.equals(VRE)){
            if(childResourceName.equals(ResourceNames.POST)){
                try {
                    return new ArrayList<>(getAllPostsByVRE(parentid));
                } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | ColumnNameNotFoundException |
                         PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            } else if(childResourceName.equals(ResourceNames.HASHTAG)){
                return new ArrayList<>(getVREHashtagsWithOccurrence(parentid));
            }
        }else if (parentResourceName.equals(ResourceNames.USER)) {
            if (childResourceName.equals(ResourceNames.POST)) {
                try {
                    return new ArrayList<>(getAllPostsByUser(parentid));
                } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | ColumnNameNotFoundException |
                         PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            } else if (childResourceName.equals(ResourceNames.NOTIFICATION_PREFERENCES)) {
                try {
                    return new ArrayList<>(getUserNotificationPreferences(parentid));
                }catch (NotificationChannelTypeNotFoundException | NotificationTypeNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }

        }else if (parentResourceName.equals(ResourceNames.APP)) {
            if (childResourceName.equals(ResourceNames.POST)) {
                try {
                    return new ArrayList<>(getAllPostsByApp(parentid));
                } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | ColumnNameNotFoundException |
                         PostIDNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return null;
    }

    @Override
    public List<IdResource> readChildOfWithParameter(String parentid, String parameterName, String parameterValue, String parentResourceName, String childResourceName){
        if(parentResourceName.equals(ResourceNames.VRE)) {
            if (childResourceName.equals(ResourceNames.POST)) {
                if (parameterName.equals(ParameterNames.RECENT_LIMIT)) {
                    try {
                        return new ArrayList<>(getRecentPostsByVRE(parentid, Integer.parseInt(parameterValue)));
                    } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException |
                             ColumnNameNotFoundException | PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                } else if (parameterName.equals(ParameterNames.HASHTAG)) {
                    try {
                        return new ArrayList<>(getVREPostsByHashtag(parentid, parameterValue));
                    } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | PostIDNotFoundException |
                             ColumnNameNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }
            } else if (childResourceName.equals(ResourceNames.HASHTAG)) {
                if (parameterName.equals(ParameterNames.TIME)) {
                    return new ArrayList<>(getVREHashtagsWithOccurrenceFilteredByTime(parentid, Long.parseLong(parameterValue)));
                }
            } else if (childResourceName.equals(ResourceNames.INVITE)){
                if (parameterName.equals(ParameterNames.STATUS)) {
                    String status = parameterValue.replace("[", "");
                    status = status.replace("]", "");
                    status = status.replace(" ", "");
                    String [] statusArray = status.split(",");
                    List<InviteStatus> inviteStatusList = new ArrayList<>();
                    for (String s: statusArray){
                        inviteStatusList.add(InviteStatus.valueOf(s));
                    }
                    try {
                        return new ArrayList<>(getInvitedEmailsByVRE(parentid, inviteStatusList));
                    } catch (InviteIDNotFoundException | InviteStatusNotFoundException e) {
                        throw new RuntimeException(e);
                    }

                } else if (parameterName.equals(ParameterNames.EMAIL)) {
                    List<Invite> toReturn = new ArrayList<>();
                    toReturn.add(isExistingInvite(parentid, parameterValue));
                    return new ArrayList<>(toReturn);
                }
            }
        }else if(parentResourceName.equals(ResourceNames.USER)){
            if (childResourceName.equals(ResourceNames.POST)){
                if (parameterName.equals(ParameterNames.RECENT_COMMENT)){
                    try {
                        return new ArrayList<>(getRecentCommentedPostsByUserAndDate(parentid, Long.parseLong(parameterValue)));
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                } else if (parameterName.equals(ParameterNames.RECENT_LIKE)){
                    return new ArrayList<>(getRecentLikedPostsByUserAndDate(parentid, Integer.parseInt(parameterValue)));

                } else if (parameterName.equals(ParameterNames.ALL_LIKE)){
                    try {
                        return new ArrayList<>(getAllLikedPostsByUser(parentid, Integer.parseInt(parameterValue)));
                    } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException |
                             ColumnNameNotFoundException | PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }

                }else if (parameterName.equals(ParameterNames.RECENT_LIMIT)){
                    try {
                        return new ArrayList<>(getRecentPostsByUser(parentid, Integer.parseInt(parameterValue)));
                    } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException |
                             ColumnNameNotFoundException | PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                } else if (parameterName.equals(ParameterNames.TIME)){
                    return new ArrayList<>(getRecentPostsByUserAndDate(parentid, Long.parseLong(parameterValue)));
                }
            } else if (childResourceName.equals(ResourceNames.NOTIFICATION)){
                if (parameterName.equals(ParameterNames.RECENT_LIMIT)){
                    try {
                        return new ArrayList<>(getAllNotificationByUser(parentid, Integer.parseInt(parameterValue)));
                    } catch (NotificationTypeNotFoundException | ColumnNameNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                } else if (parameterName.equals(ParameterNames.UNREAD)){
                    try {
                        return new ArrayList<>(getUnreadNotificationsByUser(parentid));
                    } catch (NotificationTypeNotFoundException | ColumnNameNotFoundException |
                             NotificationIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }else if (parameterName.equals(ParameterNames.NOTIFICATION_TYPE)) {
                    try {
                        List<IdResource> resources = new ArrayList<>();
                        List<NotificationChannelType> notificationChannelTypes = getUserNotificationChannels(parentid, NotificationType.valueOf(parameterName));
                        for (NotificationChannelType n: notificationChannelTypes) {
                            IdResource x = new IdResource() {
                                String notChannel = n.toString();
                                @Override
                                public String getId() {
                                    return notChannel;
                                }
                            };
                            resources.add(x);
                        }

                        return resources;
                    } catch (NotificationChannelTypeNotFoundException | NotificationTypeNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }

            }else if (childResourceName.equals(ResourceNames.COMMENT)){
                if (parameterName.equals(ParameterNames.TIME)){
                    try {
                        return new ArrayList<>(getRecentCommentsByUserAndDate(parentid, Integer.parseInt(parameterValue)));
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return null;
    }

    @Override
    public List<IdResource> readWithParameter(String parameterName, String parameterValue, String resourceName) {
        if(resourceName.equals(Post.class.getSimpleName())){
            try {
                if(parameterName.equals(PRIVACY)){
                    return new ArrayList<>(getAllPostsWithPrivacy(parameterValue));
                }
            } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException |
                     ColumnNameNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else{
            return null;
        }
        return null;
    }

    @Override
    public Boolean delete(String id, String resourceName) {
        if(resourceName.equals(Post.class.getSimpleName())){
            try {
                return deletePost(id);
            } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | PostIDNotFoundException |
                     ColumnNameNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else{
            return null;
        }
    }

    @Override
    public Boolean deleteChildOf(String parentid, String childid, String childResourceName) {
        if(childResourceName.equals(Comment.class.getSimpleName())){
            try {
                return deleteComment(parentid, childid);
            } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | PostIDNotFoundException |
                     ColumnNameNotFoundException | CommentIDNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else{
            return null;
        }
    }

    @Override
    public Boolean deleteWithParameter(String parameterName, String parameterValue, String resourceName) {
        return null;
    }

    @Override
    public Boolean BatchDeleteChildOfWithParameter(String parentid, String parameterName, String parameterValue, List<IdResource> childResource, String parentResourceName, String childResourceName){
        if(parentResourceName.equals(ResourceNames.VRE)){
            if(childResourceName.equals(ResourceNames.HASHTAG)){
                List<String> hashtags = new ArrayList<>();
                for(IdResource r: childResource){
                    Hashtag h = (Hashtag) r;
                    hashtags.add(h.getHashtag());
                }
                if(parameterName.equals(ResourceNames.POST)){
                    try {
                        return deleteHashTags(parameterValue, parentid, hashtags);
                    } catch (PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }else if(parameterName.equals(ResourceNames.COMMENT)){
                    try {
                        return deleteHashTagsComment(parameterValue, parentid, hashtags);
                    } catch (CommentIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return null;
    }
    @Override
    public Boolean deleteChildOfWithParameter(String parentid, String childid, String parameterName, String parameterValue, String parentResourceName, String childResourceName){
        if(parentResourceName.equals(ResourceNames.POST)){
            if(childResourceName.equals(ResourceNames.LIKE)){
                if(parameterName.equals(ResourceNames.USER)){
                    try {
                        return unlike(parameterValue, childid, parentid);
                    } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException |
                             ColumnNameNotFoundException | LikeIDNotFoundException | PostIDNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return null;
    }

    @Override
    public Boolean update(IdResource resource) {
        String resourceName = resource.getClass().getSimpleName();

        if(resourceName.equals(Notification.class.getSimpleName())){
            try {
                return setNotificationRead(resource.getId());
            } catch (NotificationIDNotFoundException | ColumnNameNotFoundException | NotificationTypeNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else if(resourceName.equals(Comment.class.getSimpleName())){
            try {
                return editComment((Comment) resource);
            } catch (PrivacyLevelTypeNotFoundException | PostTypeNotFoundException | ColumnNameNotFoundException |
                     CommentIDNotFoundException | PostIDNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        else{
            return null;
        }
    }

    @Override
    public Boolean updateChildOf(String parentid, String childid, IdResource childResource, String parentResourceName){
        if(parentResourceName.equals(ResourceNames.USER)) {
            String childResourceName = childResource.getClass().getSimpleName();
            if (childResourceName.equals(ResourceNames.NOTIFICATION)){
                Notification notification = (Notification) childResource;
                try {
                    if(notification.isRead()) {
                        return setAllNotificationReadByUser(parentid);
                    }
                } catch (NotificationTypeNotFoundException | ColumnNameNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        return null;
    }
    @Override
    public Boolean BatchUpdateChildOf(String parentid, List<IdResource> childResource, String parentResourceName){
        if(parentResourceName.equals(ResourceNames.USER)) {
            String childResourceName = childResource.get(0).getClass().getSimpleName();
            if (childResourceName.equals(NOTIFICATION_PREFERENCES)){
                List<NotificationPreference> notificationPreferences = new ArrayList<>();
                for(IdResource r: childResource){
                    notificationPreferences.add((NotificationPreference) r);
                }
                return setUserNotificationPreferences(parentid, notificationPreferences);
            }
        }
        return null;
    }

    @Override
    public Boolean updateWithParameter(String parameterName, String parameterValue, IdResource resource) {
        return null;
    }

    @Override
    public Boolean updateChildOfWithParameter(String parentid, String parameterName, String parameterValue, IdResource childResource, String parentResourceName){
        if(parentResourceName.equals(ResourceNames.VRE)){
            String childResourceName = childResource.getClass().getSimpleName();
            if(childResourceName.equals(ResourceNames.INVITE)){
                if(parameterName.equals(ParameterNames.EMAIL)){
                    Invite invite = (Invite) childResource;
                    try {
                        return setInviteStatus(parentid, parameterValue, invite.getStatus());
                    } catch (InviteIDNotFoundException | InviteStatusNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }

        return null;
    }


    /*
     *
     ********************** 	POSTS	***********************
     *
     */
    private boolean savePost(PostWithAttachment postWithAttachment) throws PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException, PrivacyLevelTypeNotFoundException {
        Post post = postWithAttachment.getPost();
        List<Attachment> attachments = postWithAttachment.getAttachments();
        if (attachments != null && !attachments.isEmpty())
            post.setMultiFileUpload(true);

        CqlSession session = conn.getKeyspaceSession();
        if(existRecordbyId(session, post.getKey(), POSTS, POST_ID)){
            return false;
        }
        //save into posts
        List<BoundStatement> boundStatements = insertIntoPosts(session, post);
        //save into usertimeline or apptimeline
        BoundStatement stmt2 = post.isApplicationPost()? createAppTimelineEntry(session).bind(post.getEntityId(), post.getTime().toInstant(), UUID.fromString(post.getKey()))
                                                       : createUserTimelineEntry(session).bind(post.getEntityId(), post.getTime().toInstant(), UUID.fromString(post.getKey()));
        boundStatements.add(stmt2);
        //an entry in the VRES Timeline iff vreid field is not empty
        if (post.getVreid() != null && post.getVreid().compareTo("") != 0){
            BoundStatement stmt3 = createVreTimelineEntry(session).bind(post.getVreid(), post.getTime().toInstant(), UUID.fromString(post.getKey()));
            boundStatements.add(stmt3);
        }
        //add attachments if any
        for(Attachment attachment: attachments){
            boundStatements.addAll(insertIntoAttachments(session, attachment, post.getKey()));
        }
        //execute the write
        BatchStatement writeBatch = getBatch().addAll(boundStatements);
        boolean savePostResult;
        try{
            savePostResult = session.execute(writeBatch).wasApplied();
            return savePostResult;
        }catch (Exception e){
            _log.error("Failed to save post with id {} because {} " , post.getKey(), e.getMessage());
            _log.info("Cleaning the mess");
            List<BoundStatement> cleanMess = new ArrayList<>();
            if(existRecordbyId(session, post.getKey(), POSTS, POST_ID))
                cleanMess.add(deletePostEntry(session).bind(UUID.fromString(post.getKey())));
            for (Attachment attachment : attachments) {
                if(existRecordbyId(session, attachment.getId(), ATTACHMENTS, ATTACH_ID))
                    cleanMess.add(deleteAttachmentEntry(session).bind(UUID.fromString(attachment.getId())));
            }
            BatchStatement cleanMessBatch = getBatch().addAll(cleanMess);
            boolean clean = false;
            while(!clean) {
                clean = session.execute(cleanMessBatch).wasApplied();
            }
            e.printStackTrace();
            return false;
        }
    }
    private Post readPost(String postid)
            throws PrivacyLevelTypeNotFoundException,
            PostTypeNotFoundException, PostIDNotFoundException, ColumnNameNotFoundException {
        CqlSession session = conn.getKeyspaceSession();
        Post post;
        try{
            post = findPostById(postid, session).get();

        } catch (Exception e){
            e.printStackTrace();

            return null;
        }
        return post;
    }

    private List<Post> getAllPostsWithPrivacy(String privacyLevel) throws PostTypeNotFoundException, ColumnNameNotFoundException, PrivacyLevelTypeNotFoundException {
        //possible error index
        ArrayList<Post> toReturn = new ArrayList<Post>();
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(POSTS).all()
                .whereColumn(PRIVACY).isEqualTo(QueryBuilder.bindMarker())
                .limit(20)
                .build());
        try {
            result = session.execute(stmtFind.bind(privacyLevel));

        } catch (Exception e){
            e.printStackTrace();

        }
        List<Row>rows=result.all();
        for (Row row: rows) {
            Post toAdd = readPostFromRow(row);
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH)
                toReturn.add(toAdd);
        }
        return toReturn;
    }
    private boolean deletePost(String postid) throws PostIDNotFoundException, PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException {
        Post toDelete = readPost(postid);
        CqlSession session = conn.getKeyspaceSession();
        BatchStatement writeBatch = getBatch().add(updatePostEntry(session, TYPE).bind(PostType.DISABLED.toString(), UUID.fromString(toDelete.getKey())));

        try {
            session.execute(writeBatch);

        } catch (Exception e) {
            _log.error("Delete Post ERROR for postid " + postid);

            return false;
        }
        _log.info("Delete Post OK");
        return true;
    }
    private List<IdResource> getAllChildrenByPost(String postid,String childType){
        ResultSet result = null;
        List<IdResource> toReturn = new ArrayList<>();
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(childType).all()
                .whereColumn(POST_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(UUID.fromString(postid)));
            for (Row row: result) {
                IdResource toAdd = childType.equals(LIKES) ? readLikeFromRow(row) :
                                 childType.equals(COMMENTS)? readCommentFromRow(row) :
                                 childType.equals(ATTACHMENTS)?readAttachmentFromRow(row) : null;
                if(toAdd == null) throw new Exception("No child resource named " + childType);
                toReturn.add(toAdd);
            }
        } catch (Exception e){
            e.printStackTrace();
        }
        return toReturn;
    }

    /*
     *
     ********************** 	NOTIFICATIONS	***********************
     *
     */
    private boolean saveNotification(Notification n) {
        CqlSession session = conn.getKeyspaceSession();

        List<BoundStatement> boundStatements = insertIntoNotifications(session, n);

        //an entry in the user Notifications Timeline
        BoundStatement stmt2 = createUserNotificationsEntry(session).bind(n.getUserid(), n.getTime().toInstant(), UUID.fromString(n.getKey()));
        boundStatements.add(stmt2);

        // save key in the unread notifications column family too
        BoundStatement stmt3 = createUnreadNotificationEntry(session).bind(n.getUserid(), n.getTime().toInstant(), UUID.fromString(n.getKey()));
        boundStatements.add(stmt3);
        BatchStatement writeBatch = getBatch().addAll(boundStatements);
        try{
            boolean res = session.execute(writeBatch).wasApplied();

            return res;
        }catch (Exception e){
            e.printStackTrace();
            return false;
        }
    }
    private Notification readNotification(String notificationid) throws NotificationIDNotFoundException, NotificationTypeNotFoundException, ColumnNameNotFoundException {
        Notification toReturn = new Notification();
        CqlSession session = conn.getKeyspaceSession();
        try {
            toReturn = findNotById(notificationid, session).get();

        }
        catch (Exception e) {
            e.printStackTrace();

        }
        return toReturn;
    }
    private boolean setNotificationRead(String notificationidToSet) throws NotificationIDNotFoundException, NotificationTypeNotFoundException, ColumnNameNotFoundException {
        Notification toSet = readNotification(notificationidToSet);
        if (toSet == null)
            throw new NotificationIDNotFoundException("The specified notification to set Read with id: " + notificationidToSet + " does not exist");

        CqlSession session = conn.getKeyspaceSession();
        BatchStatement writeBatch = getBatch()

                //update the entry in notifications
                .add(updateNotificationEntry(session,IS_READ).bind(true,UUID.fromString(notificationidToSet)))

                // delete the notification's key from the unread notifications column family
                .add(deleteUnreadNotEntry(session).bind(toSet.getUserid(), toSet.getTime().toInstant()));

        // execute the operations
        try {
            boolean res = session.execute(writeBatch).wasApplied();

            return res;
        } catch (Exception e) {
            _log.error("ERROR while setting Notification " + notificationidToSet + " to read.");
            return false;
        }
    }

    /*
     *
     ********************** 	COMMENTS	***********************
     *
     */

    private boolean saveComment(String postid, Comment comment) throws PostIDNotFoundException {
        Post toComment = null;
        comment.setPostid(postid);
        try {
            toComment = readPost(postid);
            if (toComment == null)
                throw new PostIDNotFoundException("Could not find Post with id " + postid + " to associate this comment", postid);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        _log.info("Writing comment : {}", comment.toString());
        CqlSession session = conn.getKeyspaceSession();
        List<BoundStatement> boundStatements = insertIntoComments(session, comment);
        BatchStatement writeBatch = getBatch().addAll(boundStatements);
        try {
            ResultSet res = session.execute(writeBatch);
            for (ExecutionInfo ex: res.getExecutionInfos()){
                _log.info("Writing comment result errors: {}", ex.getErrors());
                _log.info("Writing comment result payload: {}", ex.getIncomingPayload());
            }
            _log.info("Writing comment result executed?: {}", res.wasApplied());

        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        //update the comment count
        boolean updateCommentNoResult = updatePostCommentsCount(toComment, true);
        return updateCommentNoResult;
    }
    private Comment readComment(String commentId) throws CommentIDNotFoundException {
        Comment toReturn = null;

        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(COMMENTS).all()
                .whereColumn(COMMENT_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(UUID.fromString(commentId)));
            toReturn = readCommentFromRow(result.one());
            if (toReturn==null) {
                throw new CommentIDNotFoundException("The requested commentId: " + commentId + " is not existing");
            }

        } catch (Exception e){
            e.printStackTrace();
        }

        return toReturn;
    }
    private boolean editComment(Comment comment2Edit) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException,	CommentIDNotFoundException, PostIDNotFoundException {
        CqlSession session = conn.getKeyspaceSession();
        BatchStatement writeBatch = getBatch().add(updateCommentEntry(session, Schema.COMMENT).bind(comment2Edit.getText(), UUID.fromString(comment2Edit.getKey())))
                .add(updateCommentEntry(session, IS_EDIT).bind(true, UUID.fromString(comment2Edit.getKey())))
                .add(updateCommentEntry(session, LAST_EDIT_TIME).bind(new Date().toInstant(), UUID.fromString(comment2Edit.getKey())));

        try {
            boolean res = session.execute(writeBatch).wasApplied();

            _log.info("Comments update OK to: " + comment2Edit.getText());
            return res;
        } catch (Exception e) {

            _log.error("Comments update NOT OK ");
            return false;
        }
    }
    private boolean deleteComment(String Postid, String commentid) throws PrivacyLevelTypeNotFoundException,	PostTypeNotFoundException, ColumnNameNotFoundException,	CommentIDNotFoundException, PostIDNotFoundException {
        Post toUpdate = readPost(Postid);
        boolean updateCommentNoResult = false;

        updateCommentNoResult = updatePostCommentsCount(toUpdate, false);
        if (updateCommentNoResult) {
            CqlSession session = conn.getKeyspaceSession();
            BatchStatement writeBatch = getBatch().add(deleteCommentEntry(session).bind(UUID.fromString(commentid)));

            try {
                session.execute(writeBatch);

            } catch (Exception e) {

                _log.error("Comment Delete FAILED for " + commentid +  " from Post " + Postid);
                e.printStackTrace();
            }
            _log.trace("Comment Deleted " + commentid +  " from Post " + Postid);
        }


        return updateCommentNoResult;
    }

    /*
     *
     ********************** 	Likes	***********************
     *
     */

    private boolean like(String postid, Like like) throws PostIDNotFoundException {
        Post toLike = null;
        like.setPostid(postid);
        try {
            toLike = readPost(postid);
            if (toLike == null)
                throw new PostIDNotFoundException("Could not find post with id " + postid + " to associate this like", postid);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        if (isPostLiked(like.getUserid(), postid)) {
            _log.info("User " + like.getUserid() +  " already liked post " + postid);
            return true;
        }
        else {
            CqlSession session = conn.getKeyspaceSession();
            // Inserting data
            //an entry in the post CF
            //and an entry in the UserLikesCF
            List<BoundStatement> boundStatements = insertIntoLikes(session,like);
            BoundStatement stmt2 = createNewUserLikesEntry(session).bind(like.getUserid(), UUID.fromString(like.getKey()), UUID.fromString(like.getPostid()));
            boundStatements.add(stmt2);

            //boundStatements.forEach(stmt->writeBatch.add(stmt));
            BatchStatement writeBatch = getBatch().addAll(boundStatements);

            try {
                session.execute(writeBatch);

            } catch (Exception e) {
                e.printStackTrace();
                return false;
            }
            return updatePostLikesCount(toLike, true);
        }
    }
    private boolean unlike(String userid, String likeid, String postId) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, LikeIDNotFoundException, PostIDNotFoundException {
        Post toUpdate = readPost(postId);
        boolean updateLikeNoResult = false;

        updateLikeNoResult = updatePostLikesCount(toUpdate, false); //this remove 1 from the post CF LikeNO
        if (updateLikeNoResult) {
            CqlSession session = conn.getKeyspaceSession();
            BatchStatement writeBatch = getBatch().add(deleteLikeEntry(session).bind(UUID.fromString(likeid)))
                    .add(deleteUserLikeEntry(session).bind(userid, UUID.fromString(likeid)));
            try {
                session.execute(writeBatch);

            } catch (Exception e) {

                _log.error("Like Delete FAILED for " + likeid +  " from post " + postId);
                e.printStackTrace();
            }
            _log.trace("Unlike ok for " + likeid +  " from post " + postId);
        }
        return updateLikeNoResult;

    }

    /*
     *
     ********************** 	VRE 	***********************
     *
     */
    private List<Post> getAllPostsByVRE(String vreid) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        return getPostsByIds(getVREPostIds(vreid));
    }
    private List<Vre> getAllVREIds(){

        Set<String> ids = new HashSet<>();

        CqlSession session = conn.getKeyspaceSession();
        ResultSet result = null;
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(VRE_TIMELINE_POSTS).column(VRE_ID).all()
                .build());
        try {
            result = session.execute(stmtFind.bind());

        } catch (Exception e){
            e.printStackTrace();

        }

        List<Row>rows=result.all();
        for (Row row : rows) {
            ids.add(row.getString(VRE_ID));
        }

        List<Vre> toReturn = new ArrayList<>();
        for (String id: ids){
            Vre vre = new Vre(id);
            toReturn.add(vre);
        }

        _log.debug("VRE ids are " + ids);

        return toReturn;
    }
    private boolean savePostToVRETimeline(String vreid, Post post) throws PostIDNotFoundException {
        Post toCheck;
        try {
            toCheck = readPost(post.getKey());
            if (toCheck == null)
                throw new PostIDNotFoundException("Could not find Post with id " + post.getKey(),  post.getKey());
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        CqlSession session = conn.getKeyspaceSession();
        BatchStatement writeBatch = getBatch().add(createVreTimelineEntry(session).bind(vreid, toCheck.getTime().toInstant(), UUID.fromString(toCheck.getKey())));

        try{
            boolean res = session.execute(writeBatch).wasApplied();

            return res;
        }catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }
    private List<Hashtag> getVREHashtagsWithOccurrence(String vreid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();

        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(HASHTAGS_COUNTER).all()
                .whereColumn(VRE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(vreid));

        } catch (Exception e){
            e.printStackTrace();

        }


        //HashMap<String, Integer> toReturn = new HashMap<String, Integer> ();
        List<Hashtag> toReturn = new ArrayList<>();
        List<Row> rows = result.all();
        // Iterate rows and their columns
        for (Row row : rows) {
            Integer curValue = (int) row.getLong(COUNT);
            if (curValue > 0){
                Hashtag hashtag = new Hashtag(row.getString(Schema.HASHTAG));
                hashtag.setOccurrence(curValue);
                //toReturn.put(row.getString(Schema.HASHTAG), curValue);
                toReturn.add(hashtag);
            }

        }
        return toReturn;
    }
    private List<Hashtag> getVREHashtagsWithOccurrenceFilteredByTime(String vreid, long timestamp){
        CqlSession session = conn.getKeyspaceSession();
        ResultSet result = null;
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(HASHTAGS_COUNTER).all()
                .whereColumn(VRE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(vreid));

        } catch (Exception e){
            e.printStackTrace();

        }

        HashMap<String, Integer> check = new HashMap<String, Integer> ();
        List<Hashtag> toReturn = new ArrayList<>();
        List<Row> rows = result.all();
        // Iterate rows and their columns
        for (Row row : rows) {
            String hashtag = row.getString(Schema.HASHTAG);
            // retrieve the posts list for this hashtag
            List<Post> posts = null;
            try{
                posts = getVREPostsByHashtag(vreid, hashtag);
            }catch(Exception e){
                _log.error("Unable to retrieve the list of posts for hashtag" + hashtag + " in vre " + vreid);
                continue;
            }

            if(posts.isEmpty()){

                _log.info("There are no posts containing hashtag " + hashtag + " in  vre " + vreid);
                continue;

            }

            // retrieve the most recent one among these posts
            Collections.sort(posts, Collections.reverseOrder());

            if(posts.get(0).getTime().getTime() < timestamp){
                continue;
            }


            // else..
            int curValue = (int) row.getLong(COUNT);

            if (curValue > 0)
                check.put(row.getString(Schema.HASHTAG), curValue);
        }
        Set<String> keys = check.keySet();
        for(String key: keys){
            toReturn.add(new Hashtag(key, check.get(key)));
        }
        return toReturn;
    }
    private List<Post> getRecentPostsByVRE(String vreid, int quantity) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        ArrayList<Post> toReturn = new ArrayList<Post>();
        ArrayList<String> postIDs = getVREPostIds(vreid);
        //check if quantity is greater than user posts
        quantity = (quantity > postIDs.size()) ? postIDs.size() : quantity;

        //need them in reverse order
        for (int i = postIDs.size()-1; i >= (postIDs.size()-quantity); i--) {
            Post toAdd = readPost(postIDs.get(i));
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH) {
                toReturn.add(toAdd);
                _log.trace("Read recent Post: " + postIDs.get(i));
            } else {
                _log.trace("Read and skipped Post: " + postIDs.get(i) + " (Removed Post) .");
                quantity += 1; //increase the quantity in case of removed Post
                //check if quantity is greater than user Posts
                quantity = (quantity > postIDs.size()) ? postIDs.size() : quantity;
            }
        }
        return toReturn;
    }
    private List<Post> getVREPostsByHashtag(String vreid, String hashtag) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, PostIDNotFoundException, ColumnNameNotFoundException {
        List<Post> toReturn = new ArrayList<>();
        CqlSession session = conn.getKeyspaceSession();

        ResultSet resultPost = null;
        PreparedStatement stmtFind1 = session.prepare(QueryBuilder
                .selectFrom(HASHTAGGED_POSTS).all()
                .whereColumn(Schema.HASHTAG)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            resultPost = session.execute(stmtFind1.bind(hashtag));
        } catch (Exception e){
            e.printStackTrace();

        }

        PreparedStatement stmtFind2 = session.prepare(QueryBuilder
                .selectFrom(HASHTAGGED_COMMENTS).all()
                .whereColumn(Schema.HASHTAG)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        ResultSet resultComment = null;
        try {
            resultComment = session.execute(stmtFind2.bind(hashtag));
        } catch (Exception e){
            e.printStackTrace();

        }

        Set<String> postIds = new HashSet<>();
        // Iterate rows and their columns (post)
        List<Row>rowsPost=resultPost.all();
        for (Row row : rowsPost) {
            if (row.getString(VRE_ID).compareTo(vreid)==0)
                postIds.add(row.getUuid(POST_ID).toString());
        }
        // Iterate rows and their columns (comments)
        List<Row>rowsComment=resultComment.all();
        for (Row row : rowsComment) {
            if (row.getString(VRE_ID).compareTo(vreid)==0){
                try {
                    Comment c = readComment(row.getUuid(COMMENT_ID).toString());
                    postIds.add(c.getPostid());
                } catch (CommentIDNotFoundException e) {
                    _log.warn("Failed to fetch comment with id " + row.getString(COMMENT_ID));
                }
            }
        }
        toReturn = getPostsByIds(new ArrayList<>(postIds));
        return toReturn;
    }
    private List<Invite> getInvitedEmailsByVRE(String vreid, List<InviteStatus> status) throws InviteIDNotFoundException, InviteStatusNotFoundException{
        CqlSession session = conn.getKeyspaceSession();
        ResultSet result = null;
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(VRE_INVITES).all()
                .whereColumn(VRE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(vreid));

        } catch (Exception e){
            e.printStackTrace();

        }

        ArrayList<String> invitesIds = new ArrayList<String>();
        // Iterate rows and their columns
        List<Row>rows=result.all();
        for (Row row : rows) {
            if (status != null) {
                for (int i = 0; i < status.size(); i++) {
                    if (row.getString(STATUS).compareTo(status.get(i).toString())==0)
                        invitesIds.add(row.getUuid(INVITE_ID).toString());
                }
            }
            else {
                invitesIds.add(row.getUuid(INVITE_ID).toString());
            }
        }
        return getInvitesById(invitesIds);
    }
    private Invite isExistingInvite(String vreid, String email) {
        CqlSession session = conn.getKeyspaceSession();

        ResultSet result = null;
        PreparedStatement stmtFind1 = session.prepare(QueryBuilder
                .selectFrom(EMAIL_INVITES).all()
                .whereColumn(EMAIL)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind1.bind(email));
        } catch (Exception e){
            e.printStackTrace();

        }

        // Iterate rows and their columns
        List<Row>rows=result.all();
        for (Row row : rows) {
            if (row.getString(VRE_ID).compareTo(vreid)==0) {
                Invite invite = new Invite();
                invite.setVreid(vreid);
                invite.setInvitedEmail(email);
                invite.setKey(row.getUuid(INVITE_ID).toString());
                return invite;
            }
        }
        return null;
    }
    private boolean setInviteStatus(String vreid, String email, InviteStatus status) throws InviteIDNotFoundException, InviteStatusNotFoundException {
        String inviteid = isExistingInvite(vreid, email).getId();
        Invite toSet = readInvite(inviteid);
        if (toSet == null)
            throw new InviteIDNotFoundException("The specified invite to set with id: " + inviteid + " does not exist");

        CqlSession session = conn.getKeyspaceSession();
        BatchStatement writeBatch = getBatch().add(updateInviteEntry(session, STATUS).bind(status.toString(), UUID.fromString(inviteid)))
                .add(updateVreInviteEntry(session, STATUS).bind(status.toString(), vreid, UUID.fromString(inviteid)));
        try {
            session.execute(writeBatch);

        } catch (Exception e) {
            _log.error("ERROR while setting Invite " + inviteid + " to " + status.toString());

            return false;
        }
        _log.trace("Invite Status Set to " +  status.toString() + " OK");
        return true;
    }
    private boolean saveHashTags(String postId, String vreid, List<String> hashtags)	throws PostIDNotFoundException {
        Set<String> noduplicatesHashtags = null;
        if (hashtags != null && !hashtags.isEmpty()) {
            noduplicatesHashtags = new HashSet<String>(hashtags);
        }
        // Inserting data
        CqlSession session = conn.getKeyspaceSession();
        for (String hashtag : noduplicatesHashtags) {
            String lowerCaseHashtag = hashtag.toLowerCase();
            boolean firstInsert = session.execute((createNewHashtagTimelineEntry(session).bind(lowerCaseHashtag, UUID.fromString(postId), vreid))).wasApplied();
            boolean secondInsert = updateVREHashtagCount(vreid, lowerCaseHashtag, true);
            if (! (firstInsert && secondInsert)) {
                _log.error("saveHashTags: Could not save the hashtag(s)");

                return false;
            }
        }

        return true;
    }
    private boolean deleteHashTags(String postId, String vreid, List<String> hashtags) throws PostIDNotFoundException {
        Set<String> noduplicatesHashtags = null;
        if (hashtags != null && !hashtags.isEmpty()) {
            noduplicatesHashtags = new HashSet<String>(hashtags);
        }
        // Inserting data
        CqlSession session = conn.getKeyspaceSession();
        for (String hashtag : noduplicatesHashtags) {
            String lowerCaseHashtag = hashtag.toLowerCase();
            boolean firstDelete = session.execute(deleteHashtagEntry(session).bind(lowerCaseHashtag, UUID.fromString(postId))).wasApplied();
            boolean secondInsert = updateVREHashtagCount(vreid, lowerCaseHashtag, false);
            if (! (firstDelete && secondInsert)) {
                _log.error("deleteHashTags: Could not delete the hashtag(s)");

                return false;

            }
        }

        return true;
    }
    private boolean saveHashTagsComment(String commentId, String vreid, List<String> hashtags)	throws CommentIDNotFoundException {
        Set<String> noduplicatesHashtags = null;
        if (hashtags != null && !hashtags.isEmpty()) {
            noduplicatesHashtags = new HashSet<String>(hashtags);
        }
        // Inserting datacommentIdcommentId
        CqlSession session = conn.getKeyspaceSession();
        for (String hashtag : noduplicatesHashtags) {
            String lowerCaseHashtag = hashtag.toLowerCase();
            boolean firstInsert = session.execute(createNewHashtagCommentEntry(session).bind(hashtag, UUID.fromString(commentId), vreid)).wasApplied();
            boolean secondInsert = false;
            if(firstInsert)
                secondInsert = updateVREHashtagCount(vreid, lowerCaseHashtag, true);
            if (! (firstInsert && secondInsert)) {
                _log.error("saveHashTags: Could not save the hashtag(s)");

                return false;
            }
        }

        return true;
    }
    private boolean deleteHashTagsComment(String commentId, String vreid, List<String> hashtags) throws CommentIDNotFoundException {
        Set<String> noduplicatesHashtags = null;
        if (hashtags != null && !hashtags.isEmpty()) {
            noduplicatesHashtags = new HashSet<String>(hashtags);
        }
        // Inserting data
        CqlSession session = conn.getKeyspaceSession();
        for (String hashtag : noduplicatesHashtags) {
            String lowerCaseHashtag = hashtag.toLowerCase();
            boolean firstDelete = session.execute(deleteHashtagCommentEntry(session).bind(lowerCaseHashtag, UUID.fromString(commentId))).wasApplied();
            if(firstDelete){
                boolean secondInsert = updateVREHashtagCount(vreid, lowerCaseHashtag, false);
                if (!(firstDelete && secondInsert)) {
                    _log.error("deleteHashTags: Could not delete the hashtag(s)");

                    return false;
                }
            }else{
                _log.error("deleteHashTags: Could not delete the hashtag(s)");

                return false;
            }
        }

        return true;
    }

    /*
     *
     ********************** 	USER 	***********************
     *
     */
    private List<Post> getAllPostsByUser(String userid) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        return getPostsByIds(getUserPostIds(userid));
    }
    private List<Post> getRecentPostsByUserAndDate(String userid, long timeInMillis) throws IllegalArgumentException {
        Date now = new Date();
        if (timeInMillis > now.getTime())
            throw new IllegalArgumentException("the timeInMillis must be before today");

        ResultSet result = null;
        try {
            CqlSession session = conn.getKeyspaceSession();
            PreparedStatement stmtFind = session.prepare(QueryBuilder
                    .selectFrom(USER_TIMELINE_POSTS).all()
                    .whereColumn(USER_ID)
                    .isEqualTo(QueryBuilder.bindMarker())
                    .build());
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e) {
            e.printStackTrace();
        }
        List<Row>rows = result.all();
        List<Post> toReturn = new ArrayList<>();
        for (Row row: rows){
            Instant postTime = row.getInstant(TIMESTAMP);
            if (Date.from(postTime).getTime() > timeInMillis){
                try{
                    Post toCheck = readPost(row.getUuid(POST_ID).toString());
                    if (toCheck.getType() != PostType.DISABLED)
                        toReturn.add(toCheck);
                } catch (ColumnNameNotFoundException | PrivacyLevelTypeNotFoundException | PostIDNotFoundException |
                         PostTypeNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return toReturn;
    }
    private List<Post> getRecentPostsByUser(String userid, int quantity)	throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        ArrayList<Post> toReturn = new ArrayList<Post>();
        ArrayList<String> postIDs = getUserPostIds(userid);
        //check if quantity is greater than user posts
        quantity = (quantity > postIDs.size()) ? postIDs.size() : quantity;

        //need them in reverse order
        for (int i = postIDs.size()-1; i >= (postIDs.size()-quantity); i--) {
            Post toAdd = readPost(postIDs.get(i));
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH) {
                toReturn.add(toAdd);
                _log.trace("Read recent post: " + postIDs.get(i));
            } else {
                _log.trace("Read and skipped post: " + postIDs.get(i) + " (Removed Post)");
                quantity += 1; //increase the quantity in case of removed post
                //check if quantity is greater than user posts
                quantity = (quantity > postIDs.size()) ? postIDs.size() : quantity;
            }
        }
        return toReturn;
    }
    private List<Post> getRecentCommentedPostsByUserAndDate(String userid,
                                                            long timeInMillis) throws Exception {

        List<Post> toReturn = new ArrayList<Post>();

        Date now = new Date();
        if (timeInMillis > now.getTime())
            throw new IllegalArgumentException("the timeInMillis must be before today");

        if(userid == null || userid.isEmpty())
            throw new IllegalArgumentException("the userId parameter cannot be null/empty");

        // get the last comments by the user (it is not needed to get them sorted)
        List<Comment> lastComments = getRecentCommentsByUserAndDateBody(userid, timeInMillis, false);

        // evaluate unique posts' ids
        HashSet<String> postIds = new HashSet<String>();

        for (Comment comment : lastComments) {
            String postId = comment.getPostid();
            try{
                if(!postIds.contains(postId)){
                    postIds.add(postId);
                    toReturn.add(readPost(postId));
                }
            }catch(Exception e){
                _log.error("Unable to retrieve post with id " + postId, e);
            }
        }

        Collections.sort(toReturn, Collections.reverseOrder());
        return toReturn;
    }
    private List<Notification> getUnreadNotificationsByUser(String userid) throws NotificationTypeNotFoundException,	ColumnNameNotFoundException, NotificationIDNotFoundException {
        ArrayList<Notification> toReturn = new ArrayList<Notification>();
        ArrayList<String> notificationsIDs = getUnreadUserNotificationsIds(userid);

        //need them in reverse order
        for (int i = notificationsIDs.size()-1; i >= 0; i--) {
            try{
                toReturn.add(readNotification(notificationsIDs.get(i)));
            }catch(Exception e){
                _log.error("Unable to read notification with key " + notificationsIDs.get(i));
            }
        }

        return toReturn;
    }
    private boolean setAllNotificationReadByUser(String userid) throws NotificationTypeNotFoundException, ColumnNameNotFoundException {

        // get the list of unread notifications
        ArrayList<String> notificationsIDs = getUnreadUserNotificationsIds(userid);

        for (int i = notificationsIDs.size()-1; i >= 0; i--) {
            try{

                // set to read (and automatically remove from the unread column family)
                setNotificationRead(notificationsIDs.get(i));

            } catch (NotificationIDNotFoundException e) {
                _log.error("Could not set read notification with id =" + notificationsIDs.get(i));
            }
        }
        return true;
    }
    private List<Notification> getAllNotificationByUser(String userid, int limit) throws NotificationTypeNotFoundException,	ColumnNameNotFoundException {
        ArrayList<Notification> toReturn = new ArrayList<Notification>();
        ArrayList<String> notificationsIDs = getUserNotificationsIds(userid);
        //check if quantity is greater than user posts
        limit = (limit > notificationsIDs.size()) ? notificationsIDs.size() : limit;

        //need them in reverse order
        for (int i = notificationsIDs.size()-1; i >= (notificationsIDs.size()-limit); i--) {
            Notification toAdd = null;
            try {
                toAdd = readNotification(notificationsIDs.get(i));
                toReturn.add(toAdd);
            } catch (NotificationIDNotFoundException e) {
                _log.error("Notification not found id=" + notificationsIDs.get(i));
            }
        }
        return toReturn;
    }
    private List<Post> getRecentLikedPostsByUserAndDate(String userid,
                                                        long timeInMillis) throws IllegalArgumentException {

        List<Post> toReturn = new ArrayList<>();

        Date now = new Date();
        if (timeInMillis > now.getTime())
            throw new IllegalArgumentException("the timeInMillis must be before today");

        if(userid == null || userid.isEmpty())
            throw new IllegalArgumentException("the userId parameter cannot be null/empty");

        // get the list of liked posts
        List<String> likedPostsIdsByUser = getAllLikedPostIdsByUser(userid);

        if(likedPostsIdsByUser != null && !likedPostsIdsByUser.isEmpty()){
            for(int i = likedPostsIdsByUser.size() - 1; i >= 0; i--){
                String postid = likedPostsIdsByUser.get(i);
                try{

                    // retrieve the Post
                    Post toCheck = readPost(postid);
                    boolean isPostOk = (toCheck.getType() == PostType.TWEET || toCheck.getType() == PostType.SHARE || toCheck.getType() == PostType.PUBLISH);

                    // retrieve the like of the user for the post
                    if(isPostOk){
                        List<IdResource> likes = getAllChildrenByPost(postid, LIKES);
                        for (IdResource x : likes) {
                            Like like = (Like) x;
                            if(like.getTime().getTime() >= timeInMillis && like.getUserid().equals(userid))
                                toReturn.add(toCheck);
                        }
                    }

                }catch(Exception e){
                    _log.error("Skipped post with id " + postid, e);
                }
            }
        }

        // please check consider that if a user made like recently to an old post, well it could happen that this
        // post comes first than a newer post in the toReturn list. Thus we need to sort it.
        Collections.sort(toReturn, Collections.reverseOrder());

        return toReturn;
    }
    private boolean checkUnreadNotifications(String userid) throws NotificationTypeNotFoundException, ColumnNameNotFoundException {

        ArrayList<String> unreadNotifications = getUnreadUserNotificationsIds(userid);

        for (int i = unreadNotifications.size() - 1; i >= 0; i--) {
            Notification toAdd;
            try {
                toAdd = readNotification(unreadNotifications.get(i));
                if (toAdd.getType() != NotificationType.MESSAGE)
                    return true;
            } catch (NotificationIDNotFoundException e) {
                _log.error("Notification not found with id = " + unreadNotifications.get(i));
            }
        }

        return false;
    }
    private boolean checkUnreadMessagesNotifications(String userid) throws NotificationIDNotFoundException, NotificationTypeNotFoundException, ColumnNameNotFoundException {

        ArrayList<String> unreadNotifications = getUnreadUserNotificationsIds(userid);

        for (int i = unreadNotifications.size() - 1; i >= 0; i--) {
            Notification toAdd;
            try {
                toAdd = readNotification(unreadNotifications.get(i));
                if (toAdd.getType() == NotificationType.MESSAGE)
                    return true;
            } catch (NotificationIDNotFoundException e) {
                _log.error("Notification not found with id = " + unreadNotifications.get(i));
            }
        }

        return false;
    }
    private List<String> getAllLikedPostIdsByUser(String userid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_LIKED_POSTS).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }

        ArrayList<String> toReturn = new ArrayList<String>();
        for (Row row: result){
            toReturn.add(row.getUuid(POST_ID).toString());
        }
        return toReturn;
    } //not implemented in REST
    private List<Post> getAllLikedPostsByUser(String userid, int limit) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        ArrayList<Post> toReturn = new ArrayList<Post>();
        List<String> likedPostIDs = getAllLikedPostIdsByUser(userid);

        //check if quantity is greater than user posts
        limit = (limit > likedPostIDs.size()) ? likedPostIDs.size() : limit;

        //need them in reverse order
        for (int i = likedPostIDs.size()-1; i >= (likedPostIDs.size()-limit); i--) {
            Post toAdd = readPost(likedPostIDs.get(i));
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH) {
                toReturn.add(toAdd);
                _log.trace("Read recent post: " + likedPostIDs.get(i));
            } else {
                _log.trace("Read and skipped post: " + likedPostIDs.get(i) + " (Removed post)");
                limit += 1; //increase the quantity in case of removed post
                //check if quantity is greater than user posts
                limit = (limit > likedPostIDs.size()) ? likedPostIDs.size() : limit;
            }
        }
        return toReturn;
    }
    private List<Comment> getRecentCommentsByUserAndDate(final String userid, final long timeInMillis) throws Exception {

        final List<Comment> commentsByUser;

        Date now = new Date();
        if (timeInMillis > now.getTime())
            throw new IllegalArgumentException("the timeInMillis must be before today");

        if(userid == null || userid.isEmpty())
            throw new IllegalArgumentException("the userId parameter cannot be null/empty");


        commentsByUser = getRecentCommentsByUserAndDateBody(userid, timeInMillis, true);

        return commentsByUser;	}
    private List<NotificationChannelType> getUserNotificationChannels(String userid, NotificationType notificationType) throws NotificationChannelTypeNotFoundException, NotificationTypeNotFoundException {
        _log.info("Asking for Single Notification preference of  " + userid +  " Type: " + notificationType);
        List<NotificationChannelType> toReturn = new ArrayList<NotificationChannelType>();
        List<NotificationPreference> notificationPreferences = getUserNotificationPreferences(userid);
        NotificationPreference notificationPreference = new NotificationPreference();
        for(NotificationPreference notPref: notificationPreferences){
            if(notPref.getNotificationType().equals(notificationType.toString())){
                notificationPreference = notPref;
                break;
            }
        }

        String[] toProcess = notificationPreference.getNotificationChannelType();
        _log.info("size of user notification preferences" + toProcess.length);
        if (toProcess == null) {
            _log.info("Single Notification preference of  " + userid +  " Type: " + notificationType + " not existing ... creating default");
            return createNewNotificationType(userid, notificationType);
        }
        else if (toProcess.length == 0)
            return toReturn;
        else
            for (int i = 0; i < toProcess.length; i++) {
                toReturn.add(NotificationChannelType.valueOf(toProcess[i]));
            }
        return toReturn;
    }
    private List<NotificationPreference>  getUserNotificationPreferences(String userid) throws NotificationTypeNotFoundException, NotificationChannelTypeNotFoundException {
        _log.trace("Asking for Notification preferences of  " + userid);
        List<NotificationPreference> notificationPreferences = new ArrayList<>();
        Map<NotificationType, NotificationChannelType[]> toReturn = new HashMap<NotificationType, NotificationChannelType[]>();

        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_NOTIFICATIONS_PREFERENCES).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }

        //if there are no settings for this user create an entry and put all of them at true
        List<Row> results = result.all();
        if (results.isEmpty()) {
            _log.info("Userid " + userid + " settings not found, initiating its preferences...");
            HashMap<NotificationType, NotificationChannelType[]> toCreate = new HashMap<NotificationType, NotificationChannelType[]>();

            for (int i = 0; i < NotificationType.values().length; i++) {
                //TODO: Potential bug in NotificationType for workspace are refactored
                //create a map with all notification enabled except for workspace notifications (They start with WP_) it was the only quick way
                if (NotificationType.values()[i].toString().startsWith("WP_")) {
                    NotificationChannelType[] wpTypes = { NotificationChannelType.PORTAL };
                    toCreate.put(NotificationType.values()[i], wpTypes);
                }
                else
                    toCreate.put(NotificationType.values()[i], NotificationChannelType.values());
            }
            Set<NotificationType> keys = toCreate.keySet();
            for(NotificationType k: keys){
                NotificationChannelType[] notChannels = toCreate.get(k);
                String[] notChannelsString = new String[notChannels.length];
                for(int i = 0; i < notChannels.length; i++){
                    notChannelsString[i] = notChannels[i].toString();
                }
                NotificationPreference notPreference = new NotificationPreference(k.toString(), notChannelsString);
                notificationPreferences.add(notPreference);
            }
            setUserNotificationPreferences(userid, notificationPreferences); //commit the map

            return notificationPreferences;
        }
        else {
            _log.trace("Notification preferences Found for  " + userid);
            for (Row row: results){
                String[] channels = row.getString(PREFERENCE).split(",");
                if (channels != null && channels.length == 1 && channels[0].toString().equals("") ) { //it is empty, preference is set to no notification at all
                    toReturn.put(getNotificationType(row.getString(TYPE)), new NotificationChannelType[0]);
                } else {
                    NotificationChannelType[] toAdd = new NotificationChannelType[channels.length];
                    for (int i = 0; i < channels.length; i++) {
                        if (channels[i].compareTo("") != 0) {
                            toAdd[i] = (getChannelType(channels[i]));
                        }
                    }
                    toReturn.put(getNotificationType(row.getString(TYPE)), toAdd);
                }
            }
        }
        Set<NotificationType> keys = toReturn.keySet();
        for(NotificationType k: keys){
            NotificationChannelType[] notChannels = toReturn.get(k);
            String[] notChannelsString = new String[notChannels.length];
            for(int i = 0; i < notChannels.length; i++){
                notChannelsString[i] = notChannels[i].toString();
            }
            NotificationPreference notPreference = new NotificationPreference(k.toString(), notChannelsString);
            notificationPreferences.add(notPreference);
        }
        return notificationPreferences;
    }
    private boolean setUserNotificationPreferences(String userid, List<NotificationPreference> notificationPreferences) {
        CqlSession session = conn.getKeyspaceSession();
        List<BoundStatement> boundStatements = new ArrayList<>();
        Map<NotificationType, NotificationChannelType[]> enabledChannels = new HashMap<>();
        for(NotificationPreference notPref: notificationPreferences){
            NotificationChannelType[] notChannels = new NotificationChannelType[notPref.getNotificationChannelType().length];
            for (int i = 0; i<notPref.getNotificationChannelType().length; i++){
                notChannels[i] = NotificationChannelType.valueOf(notPref.getNotificationChannelType()[i]);
            }
            enabledChannels.put(NotificationType.valueOf(notPref.getNotificationType()), notChannels);
        }

        for (NotificationType nType : enabledChannels.keySet()) {
            String valueToInsert = "";
            _log.info("Type: " + nType.toString());
            int channelsNo = (enabledChannels.get(nType) != null) ? enabledChannels.get(nType).length : 0;
            for (int i = 0; i < channelsNo; i++) {
                _log.info(enabledChannels.get(nType)[i].toString());
                valueToInsert += NotificationChannelType.valueOf(enabledChannels.get(nType)[i].toString());
                if (i < channelsNo-1)
                    valueToInsert += ",";
            }
            if (channelsNo == 0) { //in case no channels were selected
                valueToInsert = "";
                _log.trace("No Channels selected for " + nType + " by " + userid);
            }

            boundStatements.add(createNotificationPreferenceEntry(session).bind(userid, nType.toString(), valueToInsert));
        }

        BatchStatement writeBatch = getBatch().addAll(boundStatements);
        boolean overAllresult = session.execute(writeBatch).wasApplied();
        if (overAllresult)
            _log.info("Set Notification Map for " + userid + " OK");
        else
            _log.info("Set Notification Map for " + userid + " FAILED");
        return overAllresult;
    }

    /*
     *
     ********************** 	APP 	***********************
     *
     */
    private List<Post> getAllPostsByApp(String appid) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        return getPostsByIds(getAppPostIds(appid));
    }

    /*
     *
     ********************** 	Invites	***********************
     *
     */
    private Invite readInvite(String inviteid) throws InviteIDNotFoundException, InviteStatusNotFoundException {
        Invite toReturn = null;
        CqlSession session = conn.getKeyspaceSession();
        ResultSet result = null;
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(INVITES).all()
                .whereColumn(INVITE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(UUID.fromString(inviteid)));
            toReturn = readAInviteFromRow(result.one());
            if (toReturn == null) {
                throw new InviteStatusNotFoundException("The requested inviteid: " + inviteid + " is not existing");
            }
            //toReturn = readAInviteFromRow(result.all().get(0));

        } catch (Exception e){
            e.printStackTrace();

        }
        return toReturn;
    }



    @Override
    public InviteOperationResult saveInvite(Invite invite) throws ServerException {
        if (invite == null)
            throw new NullArgumentException("Invite instance is null");
        String email = invite.getInvitedEmail();
        if (! verifyEmail(email))
            throw new ServerException("Email is not valid ->" + email);
        if (invite.getVreid() == null || invite.getVreid().equals(""))
            throw new NullArgumentException("VREId is null or empty");
        _log.debug("isExistingInvite? " + invite.getInvitedEmail() + " in " + invite.getVreid());
        if (isExistingInvite(invite.getVreid(), invite.getInvitedEmail()) != null)
            return InviteOperationResult.ALREADY_INVITED;
        _log.debug("Invite not found, proceed to save it ...");
        CqlSession session = conn.getKeyspaceSession();
        List<BoundStatement> boundStatements = initSaveInvite(invite, session);
        //an entry in the VRE Invites
        boundStatements.add(createNewVreInviteEntry(session).bind(invite.getVreid(), UUID.fromString(invite.getKey()), InviteStatus.PENDING.toString()));
        //an entry in the EMAIL Invites
        boundStatements.add(createNewEmailInviteEntry(session).bind(email, invite.getVreid(), UUID.fromString(invite.getKey())));
        BatchStatement writeBatch = getBatch().addAll(boundStatements);
        boolean result = session.execute(writeBatch).wasApplied();

        return result ? InviteOperationResult.SUCCESS : InviteOperationResult.FAILED;
    }





    /**
     * @inheritDoc
     */
    @Override
    public void closeConnection() {
        conn.closeConnection();
    }
    /*
     *
     ********************** 	Helper methods	***********************
     *
     */

    /**
     * helper method that retrieve all the posts belongin to a list of Ids
     * @param postIds
     * @return
     * @throws ColumnNameNotFoundException
     * @throws PostIDNotFoundException
     * @throws PostTypeNotFoundException
     * @throws PrivacyLevelTypeNotFoundException
     */
    private List<Post> getPostsByIds(List<String> postIds) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException, PostIDNotFoundException, ColumnNameNotFoundException {
        ArrayList<Post> toReturn = new ArrayList<Post>();
        for (String postid : postIds)  {
            Post toAdd = readPost(postid);
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH)
                toReturn.add(toAdd);
        }
        return toReturn;
    }
    /**
     * helper method that retrieve all the post Ids belonging to a user
     * @param userid  user identifier
     * @return simply return a list of user post UUID in chronological order from the oldest to the more recent
     */
    private ArrayList<String> getUserPostIds(String userid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_TIMELINE_POSTS).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }
        ArrayList<String> toReturn = new ArrayList<>();
        List<Row>rows = result.all();
        for(Row row: rows){
            try {
                String postid = row.getUuid(POST_ID).toString();
                toReturn.add(postid);
            } catch (RuntimeException e) {
                throw new RuntimeException(e);
            }
        }
        return toReturn;
    }

    /**
     * helper method that return whether the user
     * @param userid  user identifier
     * @param postid the post identifier
     * @return true if the post id liked already
     */
    private boolean isPostLiked(String userid, String postid) {

        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_LIKED_POSTS).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());

        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }
        List<Row>rows = result.all();
        for (Row row: rows){
            if (row.getUuid(POST_ID).toString().equals(postid)){
                return true;
            }
        }

        return false;
    }

    /**
     * helper method that retrieve all the post Ids belonging to an application
     * @param appid  application identifier
     * @return simply return a list of app post UUID in chronological order from the oldest to the more recent
     */
    private ArrayList<String> getAppPostIds(String appid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(APP_TIMELINE_POSTS).all()
                .whereColumn(APP_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(appid));

        } catch (Exception e){
            e.printStackTrace();

        }

        List<Row>rows=result.all();
        ArrayList<String> toReturn=new ArrayList<>();
        for(Row row: rows){
            try {
                String postid = row.getUuid(POST_ID).toString();
                toReturn.add(postid);
            } catch (RuntimeException e) {
                throw new RuntimeException(e);
            }
        }


        return toReturn;
    }

    /**
     * Private method that allows also to specify if the returned list must be sorted or not
     * @param userid the user id
     * @param timeInMillis the initial time to consider
     * @param sort a boolean value to specify if the returned list must be sorted (from the most recent to the oldest comment)
     * @return a list of comments recently made by the user
     */
    private List<Comment> getRecentCommentsByUserAndDateBody(final String userid,
                                                             final long timeInMillis, boolean sort){

        //possible error
        final List<Comment> commentsByUser = new ArrayList<Comment>();

        CqlSession session = conn.getKeyspaceSession();

        ResultSet result = null;
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(COMMENTS)
                .all()
                .build());
        try {
            result = session.execute(stmtFind.bind());

            List<Row> results = result.all();
            if (!results.isEmpty()){
                results.parallelStream().forEach( row->{
                            if(row.getString(USER_ID).equals(userid)){
                                try{
                                    Comment c = readComment(row.getUuid(COMMENT_ID).toString());
                                    Post p = readPost(c.getPostid());
                                    if(c.getTime().getTime() >= timeInMillis &&
                                            (p.getType() == PostType.TWEET || p.getType() == PostType.SHARE || p.getType() == PostType.PUBLISH))
                                        commentsByUser.add(c);
                                }catch(Exception e){
                                    _log.error("Unable to read comment with id" + row.getString(COMMENT_ID), e);
                                }
                            }
                        }
                );
            }
        } catch (Exception e){
            e.printStackTrace();

        }

        if(sort)
            Collections.sort(commentsByUser, Collections.reverseOrder());
        return commentsByUser;
    }

    /**
     * get a list of user vre post UUIDs in chronological order from the oldest to the more recent
     * @param vreid  vreid identifier (scope)
     * @return simply return a list of user vre post UUIDs in chronological order from the oldest to the more recent
     */
    private ArrayList<String> getVREPostIds(String vreid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(VRE_TIMELINE_POSTS).all()
                .whereColumn(VRE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(vreid));

        } catch (Exception e){
            e.printStackTrace();

        }

        List<Row>rows=result.all();
        ArrayList<String> toReturn=new ArrayList<>();
        for(Row row: rows){
            try {
                String postid = row.getUuid(POST_ID).toString();
                toReturn.add(postid);
            } catch (RuntimeException e) {
                throw new RuntimeException(e);
            }
        }


        return toReturn;
    }

    /**
     *
     * @param userid  user identifier
     * @return simply return a list of user notifications UUID in chronological order from the oldest to the more recent
     */
    private ArrayList<String> getUserNotificationsIds(String userid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_NOTIFICATIONS).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }

        ArrayList<String> toReturn = new ArrayList<String>();

        // Iterate rows and their columns
        List<Row>rows=result.all();
        for (Row row : rows) {
            toReturn.add(row.getUuid(NOT_ID).toString());
        }
        return toReturn;
    }
    /**
     * Return a list of not read notifications by user userid (messages as well as other notifications)
     * @param userid  user identifier
     * @return simply return a list of not read user notifications UUID in chronological order from the oldest to the more recent
     */
    private ArrayList<String> getUnreadUserNotificationsIds(String userid) {
        ResultSet result = null;
        CqlSession session = conn.getKeyspaceSession();
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(USER_NOTIFICATIONS_UNREAD).all()
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        try {
            result = session.execute(stmtFind.bind(userid));

        } catch (Exception e){
            e.printStackTrace();

        }

        ArrayList<String> toReturn = new ArrayList<String>();

        // Iterate rows and their columns
        List<Row>rows=result.all();
        for (Row row : rows) {
            toReturn.add(row.getUuid(NOT_ID).toString());
        }
        return toReturn;
    }
    /**
     * called when you add new notification types where the setting does not exist yet
     * please note: by default we set all notifications
     */
    private List<NotificationChannelType> createNewNotificationType(String userid, NotificationType notificationType) {
        List<NotificationChannelType> toReturn = new ArrayList<NotificationChannelType>();
        _log.info("Create new notification type");
        CqlSession session = conn.getKeyspaceSession();

        String valueToInsert = "";
        NotificationChannelType[] wpTypes = NotificationChannelType.values();

        for (int i = 0; i < wpTypes.length; i++) {
            valueToInsert += wpTypes[i];
            if (i < wpTypes.length-1)
                valueToInsert += ",";
            toReturn.add(wpTypes[i]); //add the new added notification type
        }

        BatchStatement writeBatch = getBatch().add(
                createNotificationPreferenceEntry(session).bind(userid, notificationType.toString(), valueToInsert)
        );
        boolean res = false;
        try{
            res = session.execute(writeBatch).wasApplied();

        } catch (Exception e){
            e.printStackTrace();

        }

        if (res) {
            _log.info("Set New Notification Setting for " + userid + " OK");
            _log.info("toreturn:" + toReturn.toString());
            return toReturn;
        }
        _log.info("empty list");
        return new ArrayList<NotificationChannelType>(); //no notification if sth fails
    }

    /**
     * helper method that retrieve all the Invites belonging to a list of Ids
     * @param inviteIds the lisf of invites UUID
     * @return all the invites belonging to a list of Ids
     * @throws InviteIDNotFoundException
     * @throws InviteStatusNotFoundException
     */
    private List<Invite> getInvitesById(List<String> inviteIds) throws InviteIDNotFoundException, InviteStatusNotFoundException {
        ArrayList<Invite> toReturn = new ArrayList<Invite>();
        for (String inviteid : inviteIds)
            toReturn.add(readInvite(inviteid));

        return toReturn;
    }
    /**
     * common part to save a invite
     * @param invite
     * @return the partial mutation batch instance
     */
    private List<BoundStatement> initSaveInvite(Invite invite, CqlSession session) {
        List<BoundStatement> boundStatements = insertIntoInvites(session, invite);
        if (invite == null)
            throw new NullArgumentException("Invite instance is null");

        return boundStatements;
    }

    /**
     * simply return an enum representing the privacy level
     * @return correct enum representing the privacy level
     * @throws NotificationChannelTypeNotFoundException
     * @throws PostTypeNotFoundException
     */
    private NotificationChannelType getChannelType(String channelName) throws NotificationChannelTypeNotFoundException {
        if (channelName.compareTo("PORTAL") == 0)
            return NotificationChannelType.PORTAL;
        else if (channelName.compareTo("EMAIL") == 0)
            return NotificationChannelType.EMAIL;
        else if (channelName.compareTo("TWITTER") == 0)
            return NotificationChannelType.TWITTER;
        else
            throw new NotificationChannelTypeNotFoundException("The Notification Channel Type was not recognized should be one of " + NotificationChannelType.values() + " asked for: " + channelName);
    }

    /**
     * simply return an enum representing the privacy level
     * @param privacyLevel .
     * @return correct enum representing the privacy level
     * @throws PostTypeNotFoundException
     */
    private static PrivacyLevel getPrivacyLevel(String privacyLevel) throws PrivacyLevelTypeNotFoundException {
        if (privacyLevel.compareTo("CONNECTION") == 0)
            return PrivacyLevel.CONNECTION;
        else if (privacyLevel.compareTo("PRIVATE") == 0)
            return PrivacyLevel.PRIVATE;
        else if (privacyLevel.compareTo("PUBLIC") == 0)
            return PrivacyLevel.PUBLIC;
        else if (privacyLevel.compareTo("VRES") == 0)
            return PrivacyLevel.VRES;
        else if (privacyLevel.compareTo("SINGLE_VRE") == 0)
            return PrivacyLevel.SINGLE_VRE;
        else if (privacyLevel.compareTo("PORTAL") == 0)
            return PrivacyLevel.PORTAL;
        else
            throw new PrivacyLevelTypeNotFoundException("The Privacy Level was not recognized should be one of " + PrivacyLevel.values() + " asked for: " + privacyLevel);
    }
    /**
     * simply return an enum representing the post type
     * @param type .
     * @return correct enum representing the post type
     * @throws PostTypeNotFoundException .
     */
    private static PostType getPostType(String type) throws PostTypeNotFoundException {
        if (type.compareTo("TWEET") == 0) {
            return PostType.TWEET;
        }
        else if (type.compareTo("JOIN") == 0) {
            return PostType.JOIN;
        }
        else if (type.compareTo("PUBLISH") == 0) {
            return PostType.PUBLISH;
        }
        else if (type.compareTo("SHARE") == 0) {
            return PostType.SHARE;
        }
        else if (type.compareTo("ACCOUNTING") == 0) {
            return PostType.ACCOUNTING;
        }
        else if (type.compareTo("DISABLED") == 0) {
            return PostType.DISABLED;
        }
        else
            throw new PostTypeNotFoundException("The post Type was not recognized should be one of " + PostType.values() + " asked for: " + type);
    }

    /**
     * simply return an enum representing the invite status type
     * @param type .
     * @return correct enum representing the post type
     * @throws InviteStatusNotFoundException .
     */
    private static InviteStatus getInviteStatusType(String type) throws InviteStatusNotFoundException {
        switch (type) {
            case "PENDING":
                return InviteStatus.PENDING;
            case "ACCEPTED":
                return InviteStatus.ACCEPTED;
            case "REJECTED":
                return InviteStatus.REJECTED;
            case "RETRACTED":
                return InviteStatus.RETRACTED;
            default:
                throw new InviteStatusNotFoundException("The Invite Status was not recognized should be one of " + InviteStatus.values() + " asked for: " + type);
        }

    }

    /**
     * simply return an enum representing the post type
     * @param type .
     * @return correct enum representing the post type
     * @throws NotificationTypeNotFoundException .
     */
    private static NotificationType getNotificationType(String type) throws NotificationTypeNotFoundException {
        if (type.compareTo("WP_FOLDER_SHARE") == 0) {
            return NotificationType.WP_FOLDER_SHARE;
        }
        else if (type.compareTo("WP_FOLDER_UNSHARE") == 0) {
            return NotificationType.WP_FOLDER_UNSHARE;
        }
        else if (type.compareTo("WP_ADMIN_UPGRADE") == 0) {
            return NotificationType.WP_ADMIN_UPGRADE;
        }
        else if (type.compareTo("WP_ADMIN_DOWNGRADE") == 0) {
            return NotificationType.WP_ADMIN_DOWNGRADE;
        }
        else if (type.compareTo("WP_FOLDER_RENAMED") == 0) {
            return NotificationType.WP_FOLDER_RENAMED;
        }
        else if (type.compareTo("WP_FOLDER_ADDEDUSER") == 0) {
            return NotificationType.WP_FOLDER_ADDEDUSER;
        }
        else if (type.compareTo("WP_FOLDER_REMOVEDUSER") == 0) {
            return NotificationType.WP_FOLDER_REMOVEDUSER;
        }
        else if (type.compareTo("WP_ITEM_DELETE") == 0) {
            return NotificationType.WP_ITEM_DELETE;
        }
        else if (type.compareTo("WP_ITEM_UPDATED") == 0) {
            return NotificationType.WP_ITEM_UPDATED;
        }
        else if (type.compareTo("WP_ITEM_NEW") == 0) {
            return NotificationType.WP_ITEM_NEW;
        }
        else if (type.compareTo("WP_ITEM_RENAMED") == 0) {
            return NotificationType.WP_ITEM_RENAMED;
        }
        else if (type.compareTo("OWN_COMMENT") == 0) {
            return NotificationType.OWN_COMMENT;
        }
        else if (type.compareTo("COMMENT") == 0) {
            return NotificationType.COMMENT;
        }
        else if (type.compareTo("MENTION") == 0) {
            return NotificationType.MENTION;
        }
        else if (type.compareTo("LIKE") == 0) {
            return NotificationType.LIKE;
        }
        else if (type.compareTo("CALENDAR_ADDED_EVENT") == 0) {
            return NotificationType.CALENDAR_ADDED_EVENT;
        }
        else if (type.compareTo("CALENDAR_UPDATED_EVENT") == 0) {
            return NotificationType.CALENDAR_UPDATED_EVENT;
        }
        else if (type.compareTo("CALENDAR_DELETED_EVENT") == 0) {
            return NotificationType.CALENDAR_DELETED_EVENT;
        }
        else if (type.compareTo("CALENDAR_ADDED_EVENT") == 0) {
            return NotificationType.CALENDAR_ADDED_EVENT;
        }
        else if (type.compareTo("CALENDAR_UPDATED_EVENT") == 0) {
            return NotificationType.CALENDAR_UPDATED_EVENT;
        }
        else if (type.compareTo("CALENDAR_DELETED_EVENT") == 0) {
            return NotificationType.CALENDAR_DELETED_EVENT;
        }
        else if (type.compareTo("MESSAGE") == 0) {
            return NotificationType.MESSAGE;
        }
        else if (type.compareTo("POST_ALERT") == 0) {
            return NotificationType.POST_ALERT;
        }
        else if (type.compareTo("REQUEST_CONNECTION") == 0) {
            return NotificationType.REQUEST_CONNECTION;
        }
        else if (type.compareTo("JOB_COMPLETED_NOK") == 0) {
            return NotificationType.JOB_COMPLETED_NOK;
        }
        else if (type.compareTo("JOB_COMPLETED_OK") == 0) {
            return NotificationType.JOB_COMPLETED_OK;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_EDIT") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_EDIT;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_VIEW") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_VIEW;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_FORWARD_STEP_COMPLETED_OWNER") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_FORWARD_STEP_COMPLETED_OWNER;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_STEP_FORWARD_PEER") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_STEP_FORWARD_PEER;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_STEP_REQUEST_TASK") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_STEP_REQUEST_TASK;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_USER_FORWARD_TO_OWNER") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_USER_FORWARD_TO_OWNER;
        }
        else if (type.compareTo("DOCUMENT_WORKFLOW_FIRST_STEP_REQUEST_INVOLVMENT") == 0) {
            return NotificationType.DOCUMENT_WORKFLOW_FIRST_STEP_REQUEST_INVOLVMENT;
        }
        else if (type.compareTo("TDM_TAB_RESOURCE_SHARE") == 0) {
            return NotificationType.TDM_TAB_RESOURCE_SHARE;
        }
        else if (type.compareTo("TDM_RULE_SHARE") == 0) {
            return NotificationType.TDM_RULE_SHARE;
        }
        else if (type.compareTo("TDM_TEMPLATE_SHARE") == 0) {
            return NotificationType.TDM_TEMPLATE_SHARE;
        }
        else if (type.compareTo("CAT_ITEM_SUBMITTED") == 0) {
            return NotificationType.CAT_ITEM_SUBMITTED;
        }
        else if (type.compareTo("CAT_ITEM_REJECTED") == 0) {
            return NotificationType.CAT_ITEM_REJECTED;
        }
        else if (type.compareTo("CAT_ITEM_PUBLISHED") == 0) {
            return NotificationType.CAT_ITEM_PUBLISHED;
        }
        else if (type.compareTo("CAT_ITEM_UPDATED") == 0) {
            return NotificationType.CAT_ITEM_UPDATED;
        }
        else if (type.compareTo("CAT_ITEM_DELETE") == 0) {
            return NotificationType.CAT_ITEM_DELETE;
        }
        else if (type.compareTo("GENERIC") == 0) {
            return NotificationType.GENERIC;
        }

        else
            throw new NotificationTypeNotFoundException("The Notification Type was not recognized should be one of " + NotificationType.values() + " asked for: " + type);
    }
    /**
     *
     * @param time in milliseconds
     * @return a Date object
     */
    private Date getDateFromTimeInMillis(String time) {
        Long timeInMillis = Long.parseLong(time);
        Calendar toSet = Calendar.getInstance();
        toSet.setTimeInMillis(timeInMillis);
        return toSet.getTime();
    }
    /**
     * update the post by incrementing or decrementing by (1) the CommentsNo
     * used when adding or removing a comment to a post
     * @param toUpdate the postId
     * @param increment set true if you want to add 1, false to subtract 1.
     */
    private boolean updatePostCommentsCount(Post toUpdate, boolean increment) {
        int newCount = 0;
        try {
            int current = Integer.parseInt(toUpdate.getCommentsNo());
            newCount = increment ? current+1 : current-1;
        }
        catch (NumberFormatException e) {
            _log.error("Comments Number found is not a number: " + toUpdate.getCommentsNo());
        }
        CqlSession session = conn.getKeyspaceSession();
        //an entry in the Post CF
        try {
            session.execute(updatePostEntry(session, COMMENTS_NO).bind((long) newCount, UUID.fromString(toUpdate.getKey())));

        } catch (Exception e) {
            _log.error("CommentsNo update NOT OK ");

            return false;
        }
        _log.info("CommentsNo update OK to: " + newCount);
        return true;
    }

    /**
     * update the post by incrementing or decrementing by (1) the LikesNo
     * used when adding or removing a comment to a post
     * @param toUpdate the postId
     * @param increment set true if you want to add 1, false to subtract 1.
     */
    private boolean updatePostLikesCount(Post toUpdate, boolean increment) {
        int newCount = 0;
        try {
            int current = Integer.parseInt(toUpdate.getLikesNo());
            newCount = increment ? current+1 : current-1;
        }
        catch (NumberFormatException e) {
            _log.error("Likes Number found is not a number: " + toUpdate.getLikesNo());
        }
        CqlSession session = conn.getKeyspaceSession();
        //an entry in the post CF
        try {
            session.execute(updatePostEntry(session, LIKES_NO).bind((long)newCount, UUID.fromString(toUpdate.getKey())));

        } catch (Exception e) {
            _log.error("LikesNo update NOT OK ");

            return false;
        }
        _log.info("LikesNo update OK to: " + newCount);
        return true;
    }

    /**
     * update the hashtag count by incrementing or decrementing it by (1)
     * used when adding or removing a hashtag in a post
     * @param vreid the vreid
     * @param hashtag the hashtag
     * @param increment set true if you want to add 1, false to subtract 1.
     */
    private boolean updateVREHashtagCount(String vreid, String hashtag, boolean increment) {
        List<Hashtag> vreHashtagsList = getVREHashtagsWithOccurrence(vreid);
        Map<String, Integer> vreHashtags = new HashMap<>();
        vreHashtagsList.forEach(x-> vreHashtags.put(x.getId(), x.getOccurrence()));

        //if the hashtag not yet exist
        int newCount = 0;

        if (!vreHashtags.containsKey(hashtag)) {
            newCount = 1;
        }
        else {
            try {
                int current = vreHashtags.get(hashtag);
                newCount = increment ? current+1 : current-1;
            }
            catch (NumberFormatException e) {
                _log.error("Hashtag Number found is not a number: " + newCount);
            }
        }
        _log.debug("Updating counter for " + hashtag + " to " + newCount);
        CqlSession session = conn.getKeyspaceSession();
        BoundStatement stmt;

        if (existRecordbyCompId(session, Schema.HASHTAG, VRE_ID, hashtag, vreid, HASHTAGS_COUNTER)){
            stmt = session.prepare(QueryBuilder.update(HASHTAGS_COUNTER)
                    .setColumn(COUNT, QueryBuilder.bindMarker())
                    .whereColumn(Schema.HASHTAG).isEqualTo(QueryBuilder.bindMarker())
                    .whereColumn(VRE_ID).isEqualTo(QueryBuilder.bindMarker())
                    .build()).bind((long)newCount, hashtag, vreid);
        }
        else{
            stmt = createNewUHashtagCounterEntry(session).bind(vreid, hashtag, (long)newCount);
        }
        BatchStatement writeBatch = getBatch().add(stmt);
        try {
            session.execute(writeBatch);

        } catch (Exception e) {
            _log.error("Hashtag Count update NOT OK ");

            return false;
        }
        _log.debug("Hashtag Count update OK to: " + newCount);
        return true;
    }
    /**
     * verify an email address
     * @param email
     * @return true or false
     */
    private boolean verifyEmail(String email) {
        boolean isValid = false;
        // TODO: validation
        return true;
        // try {
        //     InternetAddress internetAddress = new InternetAddress(email);
        //     internetAddress.validate();
        //     isValid = true;
        // } catch (AddressException e) {
        //     _log.error("Validation Exception Occurred for email: " + email);
        // }
        // return isValid;
    }






    /*
     *
     ****** Helper Functions to return prepared statements to create entries in each CF ******
     *
     */
    private static PreparedStatement createPostEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(POSTS)
                        .value(POST_ID,     QueryBuilder.bindMarker())
                        .value(LINK_HOST,    QueryBuilder.bindMarker())
                        .value(DESCRIPTION,  QueryBuilder.bindMarker())
                        .value(EMAIL,    QueryBuilder.bindMarker())
                        .value(LIKES_NO,    QueryBuilder.bindMarker())
                        .value(THUMBNAIL_URL,    QueryBuilder.bindMarker())
                        .value(LINK_DESCRIPTION,    QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(URI,    QueryBuilder.bindMarker())
                        .value(IS_APPLICATION_POST,    QueryBuilder.bindMarker())
                        .value(ENTITY_ID,    QueryBuilder.bindMarker())
                        .value(PRIVACY,    QueryBuilder.bindMarker())
                        .value(TYPE,    QueryBuilder.bindMarker())
                        .value(URI_THUMBNAIL,    QueryBuilder.bindMarker())
                        .value(VRE_ID,    QueryBuilder.bindMarker())
                        .value(MULTI_FILE_UPLOAD,    QueryBuilder.bindMarker())
                        .value(FULL_NAME,    QueryBuilder.bindMarker())
                        .value(COMMENTS_NO,    QueryBuilder.bindMarker())
                        .value(LINK_TITLE,    QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createUserTimelineEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(USER_TIMELINE_POSTS)
                        .value(USER_ID,     QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createVreTimelineEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(VRE_TIMELINE_POSTS)
                        .value(VRE_ID,     QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createAppTimelineEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(APP_TIMELINE_POSTS)
                        .value(APP_ID,     QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNotificationEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(NOTIFICATIONS)
                        .value(NOT_ID,     QueryBuilder.bindMarker())
                        .value(TYPE,    QueryBuilder.bindMarker())
                        .value(USER_ID,  QueryBuilder.bindMarker())
                        .value(SUBJECT_ID,  QueryBuilder.bindMarker())
                        .value(TIMESTAMP,  QueryBuilder.bindMarker())
                        .value(DESCRIPTION,  QueryBuilder.bindMarker())
                        .value(URI,  QueryBuilder.bindMarker())
                        .value(SENDER_ID,  QueryBuilder.bindMarker())
                        .value(SENDER_FULL_NAME,  QueryBuilder.bindMarker())
                        .value(SENDER_THUMBNAIL_URL,  QueryBuilder.bindMarker())
                        .value(IS_READ,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createUserNotificationsEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(USER_NOTIFICATIONS)
                        .value(USER_ID,     QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(NOT_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createUnreadNotificationEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(USER_NOTIFICATIONS_UNREAD)
                        .value(USER_ID,     QueryBuilder.bindMarker())
                        .value(TIMESTAMP,    QueryBuilder.bindMarker())
                        .value(NOT_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNotificationPreferenceEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(USER_NOTIFICATIONS_PREFERENCES)
                        .value(USER_ID,     QueryBuilder.bindMarker())
                        .value(TYPE,    QueryBuilder.bindMarker())
                        .value(PREFERENCE,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewCommentEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(COMMENTS)
                        .value(COMMENT_ID,     QueryBuilder.bindMarker())
                        .value(USER_ID,    QueryBuilder.bindMarker())
                        .value(FULL_NAME,  QueryBuilder.bindMarker())
                        .value(THUMBNAIL_URL,  QueryBuilder.bindMarker())
                        .value(Schema.COMMENT,  QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .value(TIMESTAMP,  QueryBuilder.bindMarker())
                        .value(IS_EDIT,  QueryBuilder.bindMarker())
                        .value(LAST_EDIT_TIME, QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewLikeEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(LIKES)
                        .value(LIKE_ID,     QueryBuilder.bindMarker())
                        .value(USER_ID,    QueryBuilder.bindMarker())
                        .value(FULL_NAME,  QueryBuilder.bindMarker())
                        .value(THUMBNAIL_URL,  QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .value(TIMESTAMP,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewUserLikesEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(USER_LIKED_POSTS)
                        .value(USER_ID,     QueryBuilder.bindMarker())
                        .value(LIKE_ID,    QueryBuilder.bindMarker())
                        .value(POST_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewHashtagTimelineEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(HASHTAGGED_POSTS)
                        .value(Schema.HASHTAG,     QueryBuilder.bindMarker())
                        .value(POST_ID,    QueryBuilder.bindMarker())
                        .value(VRE_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewHashtagCommentEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(HASHTAGGED_COMMENTS)
                        .value(Schema.HASHTAG,     QueryBuilder.bindMarker())
                        .value(COMMENT_ID,    QueryBuilder.bindMarker())
                        .value(VRE_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewEmailInviteEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(EMAIL_INVITES)
                        .value(EMAIL,     QueryBuilder.bindMarker())
                        .value(VRE_ID,  QueryBuilder.bindMarker())
                        .value(INVITE_ID,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewInviteEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(INVITES)
                        .value(INVITE_ID,     QueryBuilder.bindMarker())
                        .value(SENDER_USER_ID,    QueryBuilder.bindMarker())
                        .value(VRE_ID,  QueryBuilder.bindMarker())
                        .value(EMAIL,  QueryBuilder.bindMarker())
                        .value(CONTROL_CODE,  QueryBuilder.bindMarker())
                        .value(STATUS,  QueryBuilder.bindMarker())
                        .value(TIMESTAMP,  QueryBuilder.bindMarker())
                        .value(SENDER_FULL_NAME,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewVreInviteEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(VRE_INVITES)
                        .value(VRE_ID,     QueryBuilder.bindMarker())
                        .value(INVITE_ID,  QueryBuilder.bindMarker())
                        .value(STATUS,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewaAttachEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(ATTACHMENTS)
                        .value(ATTACH_ID,     QueryBuilder.bindMarker())
                        .value(POST_ID,    QueryBuilder.bindMarker())
                        .value(URI,  QueryBuilder.bindMarker())
                        .value(NAME,  QueryBuilder.bindMarker())
                        .value(DESCRIPTION,  QueryBuilder.bindMarker())
                        .value(URI_THUMBNAIL,  QueryBuilder.bindMarker())
                        .value(MIME_TYPE,  QueryBuilder.bindMarker())
                        .build());
    }
    private static PreparedStatement createNewUHashtagCounterEntry(CqlSession session){
        return session.prepare(
                QueryBuilder.insertInto(HASHTAGS_COUNTER)
                        .value(VRE_ID,     QueryBuilder.bindMarker())
                        .value(Schema.HASHTAG,    QueryBuilder.bindMarker())
                        .value(COUNT,  QueryBuilder.bindMarker())
                        .build());
    }

    /*
     *
     ****** Helper Functions to return prepared statements to update entries in some CF ******
     *
     */
    private static PreparedStatement updatePostEntry(CqlSession session, String colName){
        return session.prepare(QueryBuilder.update(POSTS)
                .setColumn(colName, QueryBuilder.bindMarker())
                .whereColumn(POST_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement updateInviteEntry(CqlSession session, String colName){
        return session.prepare(QueryBuilder.update(INVITES)
                .setColumn(colName, QueryBuilder.bindMarker())
                .whereColumn(INVITE_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement updateVreInviteEntry(CqlSession session, String colName){
        return session.prepare(QueryBuilder.update(VRE_INVITES)
                .setColumn(colName, QueryBuilder.bindMarker())
                .whereColumn(VRE_ID).isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(INVITE_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement updateCommentEntry(CqlSession session, String colName){
        return session.prepare(QueryBuilder.update(COMMENTS)
                .setColumn(colName, QueryBuilder.bindMarker())
                .whereColumn(COMMENT_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement updateNotificationEntry(CqlSession session, String colName){
        return session.prepare(QueryBuilder.update(NOTIFICATIONS)
                .setColumn(colName, QueryBuilder.bindMarker())
                .whereColumn(NOT_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }

    /*
     *
     ****** Helper Functions to return prepared statements to delete entries in some CF ******
     *
     */
    private static PreparedStatement deleteHashtagEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(HASHTAGGED_POSTS)
                .whereColumn(Schema.HASHTAG)
                .isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(POST_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteHashtagCommentEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(HASHTAGGED_COMMENTS)
                .whereColumn(Schema.HASHTAG)
                .isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(COMMENT_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteLikeEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(LIKES)
                .whereColumn(LIKE_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteAttachmentEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(ATTACHMENTS)
                .whereColumn(ATTACH_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deletePostEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(POSTS)
                .whereColumn(POST_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteUserLikeEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(USER_LIKED_POSTS)
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(LIKE_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteCommentEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(COMMENTS)
                .whereColumn(COMMENT_ID).isEqualTo(QueryBuilder.bindMarker())
                .build());
    }
    private static PreparedStatement deleteUnreadNotEntry(CqlSession session){
        return session.prepare(QueryBuilder.deleteFrom(USER_NOTIFICATIONS_UNREAD)
                .whereColumn(USER_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(TIMESTAMP)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
    }

    /*
     *
     ****** Helper Functions to insert objects into the DB ******
     *
     */
    private List<BoundStatement> insertIntoNotifications(CqlSession session, Notification notification){
        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(TYPE, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(USER_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(SUBJECT_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(TIMESTAMP, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(DESCRIPTION, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt6 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(URI, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt7 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(SENDER_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt8 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(SENDER_FULL_NAME, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt9 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(SENDER_THUMBNAIL_URL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt10 = session.prepare(QueryBuilder.insertInto(NOTIFICATIONS).value(NOT_ID, QueryBuilder.bindMarker()).value(IS_READ, QueryBuilder.bindMarker()).build());

        List<BoundStatement> boundStatements = new ArrayList<>();

        if(notification.getType()!=null){
            boundStatements.add(prepStmt1.bind(UUID.fromString(notification.getKey()),notification.getType().toString()));
        }

        if(notification.getUserid()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(notification.getKey()), notification.getUserid()));
        }

        if(notification.getSubjectid()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(notification.getKey()), notification.getSubjectid()));
        }
        if(notification.getTime()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(notification.getKey()),notification.getTime().toInstant()));
        }
        if(notification.getDescription()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(notification.getKey()),notification.getDescription()));
        }if(notification.getUri()!=null){
            boundStatements.add(prepStmt6.bind(UUID.fromString(notification.getKey()),notification.getUri()));
        }if(notification.getSenderid()!=null){
            boundStatements.add(prepStmt7.bind(UUID.fromString(notification.getKey()),notification.getSenderid()));
        }if(notification.getSenderFullName()!=null){
            boundStatements.add(prepStmt8.bind(UUID.fromString(notification.getKey()),notification.getSenderFullName()));
        }if(notification.getSenderThumbnail()!=null){
            boundStatements.add(prepStmt9.bind(UUID.fromString(notification.getKey()),notification.getSenderThumbnail()));
        }

        boundStatements.add(prepStmt10.bind(UUID.fromString(notification.getKey()),notification.isRead()));

        return boundStatements;
    }
    private List<BoundStatement> insertIntoComments(CqlSession session, Comment comment){
        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(USER_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(FULL_NAME, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(THUMBNAIL_URL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(Schema.COMMENT, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(POST_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt6 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(TIMESTAMP, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt7 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(IS_EDIT, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt8 = session.prepare(QueryBuilder.insertInto(COMMENTS).value(COMMENT_ID, QueryBuilder.bindMarker()).value(LAST_EDIT_TIME, QueryBuilder.bindMarker()).build());

        List<BoundStatement> boundStatements = new ArrayList<>();

        if(comment.getUserid()!=null){
            boundStatements.add(prepStmt1.bind(UUID.fromString(comment.getKey()),comment.getUserid()));
        }

        if(comment.getFullName()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(comment.getKey()), comment.getFullName()));
        }

        if(comment.getThumbnailURL()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(comment.getKey()), comment.getThumbnailURL()));
        }
        if(comment.getText()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(comment.getKey()),comment.getText()));
        }
        if(comment.getPostid()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(comment.getKey()),UUID.fromString(comment.getPostid())));
        }if(comment.getTime()!=null){
            boundStatements.add(prepStmt6.bind(UUID.fromString(comment.getKey()),comment.getTime().toInstant()));
        }
        boundStatements.add(prepStmt7.bind(UUID.fromString(comment.getKey()),comment.isEdit()));

        if(comment.getLastEditTime()!=null){
            boundStatements.add(prepStmt8.bind(UUID.fromString(comment.getKey()),comment.getLastEditTime().toInstant()));
        }

        return boundStatements;
    }
    private List<BoundStatement> insertIntoLikes(CqlSession session, Like like){

        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(LIKES).value(LIKE_ID, QueryBuilder.bindMarker()).value(USER_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(LIKES).value(LIKE_ID, QueryBuilder.bindMarker()).value(FULL_NAME, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(LIKES).value(LIKE_ID, QueryBuilder.bindMarker()).value(THUMBNAIL_URL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(LIKES).value(LIKE_ID, QueryBuilder.bindMarker()).value(POST_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(LIKES).value(LIKE_ID, QueryBuilder.bindMarker()).value(TIMESTAMP, QueryBuilder.bindMarker()).build());

        List<BoundStatement> boundStatements = new ArrayList<>();

        if(like.getUserid()!=null){
            boundStatements.add(prepStmt1.bind(UUID.fromString(like.getKey()),like.getUserid()));
        }

        if(like.getFullName()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(like.getKey()), like.getFullName()));
        }

        if(like.getThumbnailURL()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(like.getKey()), like.getThumbnailURL()));
        }

        if(like.getPostid()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(like.getKey()),UUID.fromString(like.getPostid())));
        }if(like.getTime()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(like.getKey()),like.getTime().toInstant()));
        }

        return boundStatements;
    }
    private List<BoundStatement> insertIntoInvites(CqlSession session, Invite invite){

        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(SENDER_USER_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(VRE_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(EMAIL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(CONTROL_CODE, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(STATUS, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt6 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(TIMESTAMP, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt7 = session.prepare(QueryBuilder.insertInto(INVITES).value(INVITE_ID, QueryBuilder.bindMarker()).value(SENDER_FULL_NAME, QueryBuilder.bindMarker()).build());

        List<BoundStatement> boundStatements = new ArrayList<>();

        if(invite.getSenderUserId()!=null){
            boundStatements.add(prepStmt1.bind(UUID.fromString(invite.getKey()),invite.getSenderUserId()));
        }

        if(invite.getVreid()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(invite.getKey()), invite.getVreid()));
        }

        if(invite.getInvitedEmail()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(invite.getKey()), invite.getInvitedEmail()));
        }

        if(invite.getControlCode()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(invite.getKey()),invite.getControlCode()));
        }
        if(invite.getStatus()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(invite.getKey()),invite.getStatus().toString()));
        }
        if(invite.getTime()!=null){
            boundStatements.add(prepStmt6.bind(UUID.fromString(invite.getKey()),invite.getTime().toInstant()));
        }

        if(invite.getSenderFullName()!=null){
            boundStatements.add(prepStmt7.bind(UUID.fromString(invite.getKey()),invite.getSenderFullName()));
        }

        return boundStatements;
    }
    private List<BoundStatement> insertIntoAttachments(CqlSession session, Attachment attachment, String postId){


        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(POST_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(URI, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(NAME, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(DESCRIPTION, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(URI_THUMBNAIL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt6 = session.prepare(QueryBuilder.insertInto(ATTACHMENTS).value(ATTACH_ID, QueryBuilder.bindMarker()).value(MIME_TYPE, QueryBuilder.bindMarker()).build());

        List<BoundStatement> boundStatements = new ArrayList<>();

        boundStatements.add(prepStmt1.bind(UUID.fromString(attachment.getId()),UUID.fromString(postId)));


        if(attachment.getUri()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(attachment.getId()), attachment.getUri()));
        }

        if(attachment.getName()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(attachment.getId()), attachment.getName()));
        }

        if(attachment.getDescription()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(attachment.getId()),attachment.getDescription()));
        }
        if(attachment.getThumbnailURL()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(attachment.getId()),attachment.getThumbnailURL()));
        }
        if(attachment.getMimeType()!=null){
            boundStatements.add(prepStmt6.bind(UUID.fromString(attachment.getId()),attachment.getMimeType()));
        }


        return boundStatements;
    }
    private List<BoundStatement> insertIntoPosts(CqlSession session, Post post){
        PreparedStatement prepStmt1 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(LINK_HOST, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt2 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(DESCRIPTION, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt3 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(EMAIL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt4 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(LIKES_NO, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt5 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(THUMBNAIL_URL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt6 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(LINK_DESCRIPTION, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt7 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(TIMESTAMP, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt8 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(URI, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt9 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(IS_APPLICATION_POST, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt10 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(ENTITY_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt11 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(PRIVACY, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt12 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(TYPE, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt13 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(URI_THUMBNAIL, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt14 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(VRE_ID, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt15 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(MULTI_FILE_UPLOAD, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt16 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(FULL_NAME, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt17 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(COMMENTS_NO, QueryBuilder.bindMarker()).build());
        PreparedStatement prepStmt18 = session.prepare(QueryBuilder.insertInto(POSTS).value(POST_ID, QueryBuilder.bindMarker()).value(LINK_TITLE, QueryBuilder.bindMarker()).build());
        List<BoundStatement> boundStatements = new ArrayList<>();

        if(post.getLinkHost()!=null){
            boundStatements.add(prepStmt1.bind(UUID.fromString(post.getKey()), post.getLinkHost()));
        }if(post.getDescription()!=null){
            boundStatements.add(prepStmt2.bind(UUID.fromString(post.getKey()), post.getDescription()));
        }if(post.getEmail()!=null){
            boundStatements.add(prepStmt3.bind(UUID.fromString(post.getKey()), post.getEmail()));
        }if(post.getLikesNo()!=null){
            boundStatements.add(prepStmt4.bind(UUID.fromString(post.getKey()), Long.parseLong(post.getLikesNo())));
        }if(post.getThumbnailURL()!=null){
            boundStatements.add(prepStmt5.bind(UUID.fromString(post.getKey()), post.getThumbnailURL()));
        }if(post.getLinkDescription()!=null){
            boundStatements.add(prepStmt6.bind(UUID.fromString(post.getKey()), post.getLinkDescription()));
        }if(post.getTime()!=null){
            boundStatements.add(prepStmt7.bind(UUID.fromString(post.getKey()), post.getTime().toInstant()));
        }if(post.getUri()!=null){
            boundStatements.add(prepStmt8.bind(UUID.fromString(post.getKey()), post.getUri()));
        }
        boundStatements.add(prepStmt9.bind(UUID.fromString(post.getKey()), post.isApplicationPost()));
        if(post.getEntityId()!=null){
            boundStatements.add(prepStmt10.bind(UUID.fromString(post.getKey()), post.getEntityId()));
        }if(post.getPrivacy()!=null){
            boundStatements.add(prepStmt11.bind(UUID.fromString(post.getKey()), post.getPrivacy().toString()));
        }if(post.getType()!=null){
            boundStatements.add(prepStmt12.bind(UUID.fromString(post.getKey()), post.getType().toString()));
        }if(post.getUriThumbnail()!=null){
            boundStatements.add(prepStmt13.bind(UUID.fromString(post.getKey()), post.getUriThumbnail()));
        }if(post.getVreid()!=null){
            boundStatements.add(prepStmt14.bind(UUID.fromString(post.getKey()), post.getVreid()));
        }
        boundStatements.add(prepStmt15.bind(UUID.fromString(post.getKey()), post.isMultiFileUpload()));
        if(post.getFullName()!=null){
            boundStatements.add(prepStmt16.bind(UUID.fromString(post.getKey()), post.getFullName()));
        }if(post.getCommentsNo()!=null){
            boundStatements.add(prepStmt17.bind(UUID.fromString(post.getKey()), Long.parseLong(post.getCommentsNo())));
        }if(post.getLinkTitle()!=null){
            boundStatements.add(prepStmt18.bind(UUID.fromString(post.getKey()), post.getLinkTitle()));
        }
        return boundStatements;
    }

    /*
     *
     ****** Helper Functions to fetch rows by ID from some CF ******
     *
     */
    private static Optional < Post > findPostById(String postid, CqlSession session) throws PrivacyLevelTypeNotFoundException, PostTypeNotFoundException {
        PreparedStatement stmtFindPost = session.prepare(QueryBuilder
                .selectFrom(POSTS).all()
                .whereColumn(POST_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        ResultSet rs = session.execute(stmtFindPost.bind(UUID.fromString(postid)));
        // We query by the primary key ensuring unicity
        Row record = rs.one();
        return (null != record) ? Optional.of(readPostFromRow(record)) :Optional.empty();
    }
    private static Optional < Notification > findNotById(String notid, CqlSession session) throws NotificationTypeNotFoundException {
        PreparedStatement stmtFind = session.prepare(QueryBuilder
                .selectFrom(NOTIFICATIONS).all()
                .whereColumn(NOT_ID)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        ResultSet rs = session.execute(stmtFind.bind(UUID.fromString(notid)));
        // We query by the primary key ensuring unicity
        Row record = rs.one();
        return (null != record) ? Optional.of(readNotificationFromRow(record)) :Optional.empty();
    }

    /*
     *
     ****** Helper Functions to transform DB rows into objects ******
     *
     */
    private static Attachment readAttachmentFromRow(Row record) {
        Attachment a = new Attachment();
        a.setId(Objects.requireNonNull(record.getUuid(ATTACH_ID)).toString());
        a.setUri(record.getString(URI));
        a.setName(record.getString(NAME));
        a.setDescription(record.getString(DESCRIPTION));
        a.setThumbnailURL(record.getString(URI_THUMBNAIL));
        a.setMimeType(record.getString(MIME_TYPE));
        return a;
    }
    private static Notification readNotificationFromRow(Row record) throws NotificationTypeNotFoundException {
        Notification a = new Notification();
        a.setKey(Objects.requireNonNull(record.getUuid(NOT_ID)).toString());
        a.setType(getNotificationType(Objects.requireNonNull(record.getString(TYPE))));
        a.setUserid(record.getString(USER_ID));
        a.setSubjectid(record.getString(SUBJECT_ID));
        a.setTime(Date.from(Objects.requireNonNull(record.getInstant(TIMESTAMP))));
        a.setUri(record.getString(URI));
        a.setDescription(record.getString(DESCRIPTION));
        a.setRead(record.getBoolean(IS_READ));
        a.setSenderid(record.getString(SENDER_ID));
        a.setSenderFullName(record.getString(SENDER_FULL_NAME));
        a.setSenderThumbnail(record.getString(SENDER_THUMBNAIL_URL));
        return a;
    }
    private static Post readPostFromRow(Row record) throws PostTypeNotFoundException, PrivacyLevelTypeNotFoundException {
        Post a = new Post();

        a.setKey(Objects.requireNonNull(record.getUuid(POST_ID)).toString());
        a.setType(getPostType(Objects.requireNonNull(record.getString(TYPE))));
        a.setEntityId(record.getString(ENTITY_ID));
        a.setTime(Date.from(Objects.requireNonNull(record.getInstant(TIMESTAMP))));
        a.setVreid(record.getString(VRE_ID));
        a.setUri(record.getString(URI));
        a.setUriThumbnail(record.getString(URI_THUMBNAIL));
        a.setDescription(record.getString(DESCRIPTION));
        a.setPrivacy(getPrivacyLevel(Objects.requireNonNull(record.getString(PRIVACY))));
        a.setFullName(record.getString(FULL_NAME));
        a.setEmail(record.getString(EMAIL));
        a.setThumbnailURL(record.getString(THUMBNAIL_URL));
        a.setCommentsNo(String.valueOf(record.getLong(COMMENTS_NO)));
        a.setLikesNo(String.valueOf(record.getLong(LIKES_NO)));
        a.setLinkDescription(record.getString(LINK_DESCRIPTION));
        a.setLinkTitle(record.getString(LINK_TITLE));
        a.setLinkHost(record.getString(LINK_HOST));
        a.setApplicationPost(record.getBoolean(IS_APPLICATION_POST));
        a.setMultiFileUpload(record.getBoolean(MULTI_FILE_UPLOAD));

        return a;
    }
    private static Like readLikeFromRow(Row record) {
        Like a = new Like();
        a.setKey(Objects.requireNonNull(record.getUuid(LIKE_ID)).toString());
        a.setUserid(record.getString(USER_ID));
        a.setTime(Date.from(Objects.requireNonNull(record.getInstant(TIMESTAMP))));
        a.setPostid(Objects.requireNonNull(record.getUuid(POST_ID)).toString());
        a.setFullName(record.getString(FULL_NAME));
        a.setThumbnailURL(record.getString(THUMBNAIL_URL));
        return a;
    }
    private static Comment readCommentFromRow(Row record) {
        Comment a = new Comment();
        a.setKey(Objects.requireNonNull(record.getUuid(COMMENT_ID)).toString());
        a.setUserid(record.getString(USER_ID));
        a.setTime(Date.from(Objects.requireNonNull(record.getInstant(TIMESTAMP))));
        a.setPostid(Objects.requireNonNull(record.getUuid(POST_ID)).toString());
        a.setFullName(record.getString(FULL_NAME));
        a.setThumbnailURL(record.getString(THUMBNAIL_URL));
        a.setText(record.getString(Schema.COMMENT));
        a.setEdit(!record.isNull(IS_EDIT) && record.getBoolean(IS_EDIT));
        a.setLastEditTime(record.isNull(LAST_EDIT_TIME)? null : Date.from(Objects.requireNonNull(record.getInstant(LAST_EDIT_TIME))));
        return a;
    }
    private static Invite readAInviteFromRow(Row record) throws InviteStatusNotFoundException {
        Invite a = new Invite();
        a.setKey(Objects.requireNonNull(record.getUuid(INVITE_ID)).toString());
        a.setSenderUserId(record.getString(SENDER_USER_ID));
        a.setVreid(record.getString(VRE_ID));
        a.setInvitedEmail(record.getString(EMAIL));
        a.setControlCode(record.getString(CONTROL_CODE));
        a.setStatus(getInviteStatusType(Objects.requireNonNull(record.getString(STATUS))));
        a.setTime(Date.from(Objects.requireNonNull(record.getInstant(TIMESTAMP))));
        a.setSenderFullName(record.getString(SENDER_FULL_NAME));
        return a;
    }

    /*
     *
     ****** Helper Functions to check record existence by id ******
     *
     */
    private static boolean existRecordbyId(CqlSession session, String id, String tableName, String colName) {
        PreparedStatement stmt = session.prepare(QueryBuilder
                .selectFrom(tableName).column(colName)
                .whereColumn(colName)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        return session.execute(stmt.bind(id)).getAvailableWithoutFetching() > 0;
    }

    private static boolean existRecordbyCompId(CqlSession session, String col1, String col2, String id1, String id2, String tableName) {
        PreparedStatement stmt = session.prepare(QueryBuilder
                .selectFrom(tableName).all()
                .whereColumn(col1)
                .isEqualTo(QueryBuilder.bindMarker())
                .whereColumn(col2)
                .isEqualTo(QueryBuilder.bindMarker())
                .build());
        return session.execute(stmt.bind(id1, id2)).getAvailableWithoutFetching() > 0;
    }

    /*
     *
     ********************** 	Helper function to get batch statement	***********************
     *
     */
    private static BatchStatement getBatch(){
        return BatchStatement.builder(BatchType.LOGGED).build();
    }
}

    /*private RangePosts getRecentPostsByVREAndRange(String vreid, int from, int quantity) throws IllegalArgumentException, PrivacyLevelTypeNotFoundException,	PostTypeNotFoundException, ColumnNameNotFoundException, PostIDNotFoundException {
        if (from < 1) {
            throw new IllegalArgumentException("From must be greather than 0");
        }
        ArrayList<Post> postsToReturn = new ArrayList<Post>();
        ArrayList<String> postIDs = getVREPostIds(vreid);

        //if from is greater than posts size return empty
        if (from >=  postIDs.size()) {
            _log.warn("The starting point of the range is greather than the total number of posts for this timeline: " + from + " >= " + postIDs.size());
            return new RangePosts();
        }

        int rangeStart = postIDs.size()-from;
        int rangeEnd = rangeStart-quantity;

        //check that you reached the end
        if (rangeEnd<1)
            rangeEnd = 0;

        _log.debug("BEFORE starting Point=" + rangeStart + " rangeEnd= " + rangeEnd);
        //need them in reverse order
        int howMany = from;
        for (int i = rangeStart; i > rangeEnd; i--) {
            Post toAdd = readPost(postIDs.get(i));
            if (toAdd.getType() == PostType.TWEET || toAdd.getType() == PostType.SHARE || toAdd.getType() == PostType.PUBLISH) {
                postsToReturn.add(toAdd);
                _log.trace("Read recent post, i=" + i + " id= " + postIDs.get(i));
            } else {
                _log.trace("Read and skipped post, i=" + i + " id=: " + postIDs.get(i) + " (Removed post) .");
                rangeEnd -= 1; //increase the upTo in case of removed post
                //check if quantity is greater than user posts
                rangeEnd = (rangeEnd > 0) ? rangeEnd : 0;
            }
            howMany++;
        }
        _log.debug("AFTER: starting Point==" + rangeStart + " rangeEnd= " + rangeEnd);
        return new RangePosts(howMany+1, postsToReturn);
    }

    private List<Notification> getRangeNotificationsByUser(String userid,int from, int quantity) throws NotificationTypeNotFoundException, ColumnNameNotFoundException, NotificationIDNotFoundException {
        if (from < 1) {
            throw new IllegalArgumentException("From must be greather than 0");
        }
        ArrayList<Notification> toReturn = new ArrayList<Notification>();
        ArrayList<String> notificationsIDs = getUserNotificationsIds(userid);

        //if from is greater than posts size return empty
        if (from >=  notificationsIDs.size()) {
            _log.warn("The starting point of the range is greather than the total number of posts for this timeline: " + from + " >= " + notificationsIDs.size());
            return new  ArrayList<Notification>();
        }

        int rangeStart = notificationsIDs.size()-from;
        int rangeEnd = rangeStart-quantity;

        //check that you reached the end
        if (rangeEnd<1)
            rangeEnd = 0;

        _log.debug("BEFORE starting Point=" + rangeStart + " rangeEnd= " + rangeEnd);
        //need them in reverse order
        for (int i = rangeStart; i > rangeEnd; i--) {
            Notification toAdd = readNotification(notificationsIDs.get(i));
            toReturn.add(toAdd);
        }
        return toReturn;
    }
    */

