package org.gcube.spatial.data.gis.wms;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
     
    /**
     * The Class WmsUrlValidator.
     *
     * @author Francesco Mangiacrapa francesco.mangiacrapa@isti.cnr.it
     * Dec 4, 2015
     */
    public class WmsRequestValidator {
     
    	private HashMap<String, String> parametersValue = new HashMap<String, String>();
    	private String wmsRequest;
    	private String baseWmsServiceUrl;
    	private String wmsParameters;
    	private String wmsNotStandardParameters = "";
    	private Map<String, String> mapWmsNotStandardParams;
     
    	public static Logger logger = LoggerFactory.getLogger(WmsRequestValidator.class);
     
    	/**
    	 * Instantiates a new wms url validator.
    	 *
    	 * @param wmsRequest the wms request
    	 */
    	public WmsRequestValidator(String wmsRequest){
    		this.wmsRequest = wmsRequest;
    		int indexStart = wmsRequest.indexOf("?");
    		if(indexStart==-1){
    			this.baseWmsServiceUrl = wmsRequest;
    			this.wmsParameters = null;
    		}else{
    			this.baseWmsServiceUrl=wmsRequest.substring(0, indexStart);
    			this.baseWmsServiceUrl.trim();
    			this.wmsParameters = wmsRequest.substring(indexStart+1, this.wmsRequest.length());
    			this.wmsParameters.trim();
    		}
    	}
     
    	/**
    	 * Parses the wms request.
    	 *
    	 * @param returnEmptyParameter - return empty parameter (e.g styles=)
    	 * @param fillEmptyParameterAsDefaultValue - fill empty parameter as default value
    	 * @return parsed wms request
    	 * @throws Exception 
    	 */
    	public String parseWmsRequest(boolean returnEmptyParameter, boolean fillEmptyParameterAsDefaultValue) throws Exception{
     
    		if(wmsParameters==null || wmsParameters.isEmpty()){
    			String msg = "IT IS NOT POSSIBLE TO PARSE WMS URL, 'WMS PARAMETERS' not found!";
    			throw new Exception(msg);
    		}
     
    		for (WmsParameters wmsParam : WmsParameters.values()) {
     
    			if(wmsParam.equals(WmsParameters.BBOX)){
    				String value = validateValueOfParameter(WmsParameters.BBOX, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.FORMAT)){
    				String value = validateValueOfParameter(WmsParameters.FORMAT, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.HEIGHT)){
    				String value =  validateValueOfParameter( WmsParameters.HEIGHT, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.CRS)){
    				String crs = validateValueOfParameter(WmsParameters.CRS, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), crs);
    			}
     
    			if(wmsParam.equals(WmsParameters.WIDTH)){
    				String value = validateValueOfParameter(WmsParameters.WIDTH, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.REQUEST)){
    				String value = validateValueOfParameter(WmsParameters.REQUEST, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.SERVICE)){
    				String value = validateValueOfParameter(WmsParameters.SERVICE, wmsParameters,fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.SRS)){
    				String value = validateValueOfParameter(WmsParameters.SRS, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), value);
    			}
     
    			if(wmsParam.equals(WmsParameters.STYLES)){
    				String styles = validateValueOfParameter(WmsParameters.STYLES, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), styles);
    			}
     
    			if(wmsParam.equals(WmsParameters.VERSION)){
    				String version = validateValueOfParameter(WmsParameters.VERSION, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), version);
    			}
     
    			if(wmsParam.equals(WmsParameters.LAYERS)){
    				String layers = validateValueOfParameter(WmsParameters.LAYERS, wmsParameters, fillEmptyParameterAsDefaultValue);
    				parametersValue.put(wmsParam.getParameter(), layers);
    			}
    		}
     
    		String parsedWmsRequest = baseWmsServiceUrl+"?";
     
    		String[] params = wmsParameters.split("&");
     
    		//CREATING MAP TO RETURN WMS PARAMETERS NOT STANDARD
    		mapWmsNotStandardParams = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
     
    		for (String param : params) {
    			String key = param.substring(0, param.indexOf("="));
    			String value = param.substring(param.indexOf("=")+1, param.length());
    			mapWmsNotStandardParams.put(key, value);
    		}
     
    		//CREATE WMS REQUEST
    		for (String key : parametersValue.keySet()) {
     
    			String value = parametersValue.get(key);
    			if(returnEmptyParameter && value.isEmpty()){
    				parsedWmsRequest+=key+"="+value;
    				//not add the parameter
    			}else{
    				parsedWmsRequest+=key+"="+value;
    				parsedWmsRequest+="&";
    				String exist = mapWmsNotStandardParams.get(key);
     
    				if(exist!=null)
    					mapWmsNotStandardParams.remove(key); //REMOVE WMS STANDARD PARAMETER FROM MAP
    			}
    		}
     
    		for (String key : mapWmsNotStandardParams.keySet()) {
    			wmsNotStandardParameters+=key+"="+mapWmsNotStandardParams.get(key) + "&";
    		}
     
    		if(wmsNotStandardParameters.length()>0)
    			wmsNotStandardParameters = wmsNotStandardParameters.substring(0, wmsNotStandardParameters.length()-1); //REMOVE LAST &
     
    		String fullWmsUrlBuilded;
     
    		if(!wmsNotStandardParameters.isEmpty()){
    			fullWmsUrlBuilded = parsedWmsRequest +  wmsNotStandardParameters; //remove last &
    		}else{
    			fullWmsUrlBuilded = parsedWmsRequest.substring(0, parsedWmsRequest.length()-1); //remove last &
    		}
     
    		return fullWmsUrlBuilded;
    	}
     
    	/**
    	 * @return the wmsRequest
    	 */
    	public String getWmsRequest() {
    		return wmsRequest;
    	}
     
    	/**
    	 * @return the baseWmsServiceUrl
    	 */
    	public String getBaseWmsServiceUrl() {
    		return baseWmsServiceUrl;
    	}
     
    	/**
    	 * Gets the wms not standard parameters.
    	 *
    	 * @return the wms not standard parameters
    	 */
    	public String getWmsNotStandardParameters() {
    		return wmsNotStandardParameters;
    	}
     
     
    	/**
    	 * Gets the value of parsed wms parameter.
    	 *
    	 * @param parameter the parameter
    	 * @return the value of parsed wms parameter parsed from wms request.
    	 */
    	public String getValueOfParsedWMSParameter(WmsParameters parameter){
    		return parametersValue.get(parameter.getParameter());
    	}
     
     
    	/**
    	 * Validate value of parameter.
    	 *
    	 * @param wmsParam the wms param
    	 * @param valueOfParameter the value of parameter
    	 * @param fillEmptyParameterAsDefaultValue the fill empty parameter as default value
    	 * @return the string
    	 */
    	public static String validateValueOfParameter(WmsParameters wmsParam, String valueOfParameter, boolean fillEmptyParameterAsDefaultValue){
     
    		String value = getValueOfParameter(wmsParam, valueOfParameter);
    		if(fillEmptyParameterAsDefaultValue && (value==null || value.isEmpty())){
    			value = wmsParam.getValue();
    		}
    		return value;
    	}
     
    	/**
    	 * Gets the value of parameter.
    	 *
    	 * @param wmsParam the wms param
    	 * @param wmsUrlParameters the url wms parameters
    	 * @return the value of parameter
    	 */
    	public static String getValueOfParameter(WmsParameters wmsParam, String wmsUrlParameters){
    		logger.trace("getValueOfParameter: "+wmsParam +" into "+wmsUrlParameters);
    		int index = wmsUrlParameters.toLowerCase().indexOf(wmsParam.getParameter().toLowerCase());
    		String value = "";
    		if(index > -1){
     
    			int start = index + wmsParam.getParameter().length()+1; //add +1 for char '='
    			String sub = wmsUrlParameters.substring(start, wmsUrlParameters.length()); 
    			int indexOfSeparator = sub.indexOf("&");
    			int end = indexOfSeparator!=-1?indexOfSeparator:sub.length();
    			value = sub.substring(0, end);
    		}
    		logger.trace("returning value: "+value);
    		return value;
    	}
     
     
    	/**
    	 * Sets the value of parameter.
    	 *
    	 * @param wmsParam the wms param
    	 * @param wmsUrlParameters the wms url parameters
    	 * @param newValue the new value
    	 * @param addIfNotExists add the parameter if not exists
    	 * @return the string
    	 */
    	public static String setValueOfParameter(WmsParameters wmsParam, String wmsUrlParameters, String newValue, boolean addIfNotExists){
    		logger.trace("setValueOfParameter: "+wmsParam +" into "+wmsUrlParameters);
    		String toLowerWmsUrlParameters = wmsUrlParameters.toLowerCase();
    		String toLowerWmsParam = wmsParam.getParameter().toLowerCase();
     
    		int index = toLowerWmsUrlParameters.indexOf(toLowerWmsParam+"="); //+ "=" SECURE TO BE PARAMETER 
    //		logger.trace("start index of "+wmsParam+ " is: "+index);
    		if(index > -1){		
    			int indexStartValue = index + toLowerWmsParam.length()+1; //add +1 for char '='
    			int indexOfSeparator = toLowerWmsUrlParameters.indexOf("&", indexStartValue); //GET THE FIRST "&" STARTING FROM INDEX VALUE
    //			logger.trace("indexOfSeparator index of "+wmsParam+ " is: "+indexOfSeparator);
    			int indexEndValue = indexOfSeparator!=-1?indexOfSeparator:toLowerWmsUrlParameters.length();
    //			logger.trace("end: "+indexEndValue);
    			return wmsUrlParameters.substring(0, indexStartValue) + newValue +wmsUrlParameters.substring(indexEndValue, wmsUrlParameters.length());
    		}else if (addIfNotExists){
    			wmsUrlParameters+="&"+wmsParam.getParameter()+"="+newValue;
    		}
    		logger.trace("returning value: "+wmsUrlParameters);
    		return wmsUrlParameters;
    	}
     
    	/**
    	 * Gets the styles as list.
    	 *
    	 * @return the styles as list
    	 */
    	public List<String> getStylesAsList() {
     
    		List<String> listStyles = new ArrayList<String>();
    		String styles = getValueOfParsedWMSParameter(WmsParameters.STYLES);
     
    		if(styles!=null && !styles.isEmpty()){
     
    			String[] arrayStyle = styles.split(",");
    			for (String style : arrayStyle) {
    				if(style!=null && !style.isEmpty())
    					listStyles.add(style);
    			}
    		}	
    		return listStyles;
    	}
     
    	/**
    	 * Gets the map wms not standard params.
    	 *
    	 * @return the mapWmsNotStandardParams
    	 */
    	public Map<String, String> getMapWmsNotStandardParams() {
    		return mapWmsNotStandardParams;
    	}
     
    }