package org.gcube.vremanagement.executor.api;

import javax.jws.WebService;
import javax.jws.soap.SOAPBinding;
import javax.jws.soap.SOAPBinding.ParameterStyle;
import javax.jws.soap.SOAPBinding.Style;
import javax.jws.soap.SOAPBinding.Use;

import org.gcube.vremanagement.executor.api.types.LaunchParameter;
import org.gcube.vremanagement.executor.exception.ExecutorException;
import org.gcube.vremanagement.executor.exception.InputsNullException;
import org.gcube.vremanagement.executor.exception.LaunchException;
import org.gcube.vremanagement.executor.exception.PluginInstanceNotFoundException;
import org.gcube.vremanagement.executor.exception.PluginNotFoundException;
import org.gcube.vremanagement.executor.plugin.PluginState;

/**
 * Service Endpoint Interface
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
@WebService(serviceName = SmartExecutor.WEB_SERVICE_SERVICE_NAME, targetNamespace=SmartExecutor.TARGET_NAMESPACE)
@SOAPBinding(style = Style.DOCUMENT, use=Use.LITERAL)
public interface SmartExecutor {
	
	public static final String TARGET_NAMESPACE = "http://gcube-system.org/";
	public static final String WEB_SERVICE_SERVICE_NAME = "gcube/vremanagement/smart-executor";
	
	/**
	 * Launch the plugin identified by the name provided as parameters
	 * with the provided inputs. Inputs cannot be null. If you need to launch 
	 * an execution with no inputs provide an empty Map.
	 * This method return as soon as the plugin has been launched.
	 * The execution is made in a separated Thread. 
	 * @param launchParameter which contains the name of the plugin to launch 
	 * and the input to be provided to plugin to run and the scheduling strategy.
	 * @return the UUID execution identifier as String which identify the 
	 * execution of the plugin.
	 * @throws InputsNullException if {@link LaunchParameter} contains null 
	 * inputs.
	 * @throws PluginNotFoundException if {@link LaunchParameter} contains a 
	 * name of a plugin which is not available on classpath
	 * @throws LaunchException if an error occurs trying to instantiate and/or
	 * launch the plugin execution
	 * @throws ExecutorException if any other undefined error occur
	 */
	@SOAPBinding(parameterStyle=ParameterStyle.WRAPPED)
	public String launch(LaunchParameter launchParameter) throws 
		InputsNullException, PluginNotFoundException, 
		LaunchException, ExecutorException;
	
	/**
	 * The method use the provided UUID execution identifier as String to 
	 * unschedule a previous Scheduled Task and stop the current execution 
	 * (if any).
	 * @param executionIdentifier UUID as String which identify the execution
	 * @return return true if the current execution has been correctly 
	 * unscheduled. False otherwise.
	 * @throws Exception if there is no Scheduled Task identified by the 
	 * provided UUID execution identifier as String
	 */
	@SOAPBinding(parameterStyle=ParameterStyle.WRAPPED)
	public boolean unSchedule(String executionIdentifier) throws ExecutorException;
	
	
	/**
	 * The method use the provided UUID as String to retrieve the status of the 
	 * associated execution 
	 * @param executionIdentifier UUID as String which identify the execution
	 * @return {@link PluginState} which contains the state of the execution
	 * @throws Exception if there is no execution identified by the provided 
	 * UUID execution identifier as String
	 */
	@SOAPBinding(parameterStyle=ParameterStyle.WRAPPED)
	public PluginState getState(String executionIdentifier) 
			throws PluginInstanceNotFoundException, ExecutorException;
	
	/**
	 * The method use the provided UUID as String and the iteration number
	 * to retrieve the status of the associated execution 
	 * @param executionIdentifier UUID as String which identify the execution
	 * @param iterationNumber iteration number
	 * @return {@link PluginState} which contains the state of the execution
	 * @throws Exception if there is no execution identified by the provided 
	 * UUID execution identifier as String
	 */
	@SOAPBinding(parameterStyle=ParameterStyle.WRAPPED)
	public PluginState getIterationState(String executionIdentifier, int iterationNumber) 
			throws PluginInstanceNotFoundException, ExecutorException;
	
}
