/**
 * 
 */
package org.gcube.vremanagement.executor.api.types;

import java.util.Calendar;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.quartz.CronExpression;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
@XmlRootElement()
@XmlAccessorType(XmlAccessType.FIELD)
public class Scheduling implements Comparable<Scheduling> {
	
	/**
	 * CRON like expression for a repetitive task.
	 * This field is not valid when using delay
	 */
	@XmlElement
	protected String cronExpression;
	
	/**
	 * Delay between subsequent execution in seconds.
	 * This field is not valid when using cronExpression
	 */
	@XmlElement
	protected Integer delay;
	
	/**
	 * Indicates the number of times the scheduling pattern must be applied.
	 * 0 means indefinitely.
	 */
	@XmlElement
	protected int schedulingTimes;
	
	/**
	 * The first instant when the scheduling can start
	 */
	@XmlElement
	protected Long firstStartTime; // O or null means immediately
	
	/**
	 * Time at which the Trigger will no longer fire even if it's schedule 
	 * has remaining repeats.
	 */
	@XmlElement
	protected Long endTime; // O or null means never
	
	/**
	 * When using cronExpression run the subsequent task only if the previous 
	 * are terminated otherwise this execution is discarded and the subsequent 
	 * execution will start when fired by the the next scheduling.
	 * The discarded execution is counted in the total number of executions
	 * happened.
	 */
	@XmlElement
	protected boolean previuosExecutionsMustBeCompleted;
	
	
	protected void init(CronExpression cronExpression, Integer delay, int schedulingTimes, Long firstStartTime, Long endTime, boolean previuosExecutionsMustBeCompleted){
		if(cronExpression!=null){
			this.cronExpression = cronExpression.getCronExpression();
		}else{
			this.cronExpression = null;
		}
		this.delay = delay;
		this.schedulingTimes = schedulingTimes;
		this.firstStartTime = firstStartTime;
		this.endTime = endTime;
		this.previuosExecutionsMustBeCompleted = previuosExecutionsMustBeCompleted;
	}
	
	protected Scheduling(){}
	
	public Scheduling(CronExpression cronExpression) {
		init(cronExpression, null, 0, null, null, false);
	}
	
	public Scheduling(CronExpression cronExpression, boolean previuosExecutionsMustBeCompleted) {
		init(cronExpression, null, 0, null, null, previuosExecutionsMustBeCompleted);
	}
	
	public Scheduling(CronExpression cronExpression, int schedulingTimes) {
		init(cronExpression, null, schedulingTimes, null, null, false);
	}
	
	public Scheduling(CronExpression cronExpression, int schedulingTimes, boolean previuosExecutionsMustBeCompleted ) {
		init(cronExpression, null, schedulingTimes, null, null, previuosExecutionsMustBeCompleted);
	}
	
	public Scheduling(CronExpression cronExpression, int schedulingTimes, Calendar firstStartTime, Calendar endTime) {
		init(cronExpression, null, schedulingTimes, firstStartTime.getTimeInMillis(), endTime.getTimeInMillis(), false);
	}
	
	public Scheduling(CronExpression cronExpression, int schedulingTimes, Calendar firstStartTime, Calendar endTime, boolean previuosExecutionsMustBeCompleted) {
		init(cronExpression, null, schedulingTimes, firstStartTime.getTimeInMillis(), endTime.getTimeInMillis(), previuosExecutionsMustBeCompleted);
	}
	
	
	public Scheduling(int delay) {
		init(null, delay, 0, null, null, false);
	}
	
	public Scheduling(int delay, boolean previuosExecutionsMustBeCompleted) {
		init(null, delay, 0, null, null, previuosExecutionsMustBeCompleted);
	}
	
	public Scheduling(int delay, int schedulingTimes) {
		init(null, delay, schedulingTimes, null, null, false);
	}
	
	public Scheduling(int delay, int schedulingTimes, boolean previuosExecutionsMustBeCompleted ) {
		init(null, delay, schedulingTimes, null, null, previuosExecutionsMustBeCompleted);
	}
	
	public Scheduling(int delay, int schedulingTimes, Calendar firstStartTime, Calendar endTime) {
		init(null, delay, schedulingTimes, firstStartTime.getTimeInMillis(), endTime.getTimeInMillis(), false);
	}
	
	public Scheduling(int delay, int schedulingTimes, Calendar firstStartTime, Calendar endTime, boolean previuosExecutionsMustBeCompleted) {
		init(null, delay, schedulingTimes, firstStartTime.getTimeInMillis(), endTime.getTimeInMillis(), previuosExecutionsMustBeCompleted);
	}

	/**
	 * @return the cronExpression
	 */
	public String getCronExpression() {
		return cronExpression;
	}

	/**
	 * @return the delay
	 */
	public Integer getDelay() {
		return delay;
	}

	/**
	 * @return the schedulingTimes
	 */
	public int getSchedulingTimes() {
		return schedulingTimes;
	}

	/**
	 * @return the previuosExecutionMustBeCompleted
	 */
	public boolean mustPreviousExecutionsCompleted() {
		return previuosExecutionsMustBeCompleted;
	}

	/**
	 * @return the firtStartTime
	 */
	public Long getFirtStartTime() {
		return firstStartTime;
	}

	/**
	 * @return the firstStartTime
	 */
	public Long getFirstStartTime() {
		return firstStartTime;
	}

	/**
	 * @param firstStartTime the firstStartTime to set
	 */
	public void setFirstStartTime(Long firstStartTime) {
		this.firstStartTime = firstStartTime;
	}

	/**
	 * @return the endTime
	 */
	public Long getEndTime() {
		return endTime;
	}
	
	public String toString(){
		return String.format("CronExpression %s, Delay %d, SchedulingTimes %d, FirstStartTime %d, EndTime %d, PreviuosExecutionsMustBeCompleted %b", 
				cronExpression, delay, schedulingTimes, firstStartTime, endTime, previuosExecutionsMustBeCompleted);
	}

	/** {@inheritDoc}} */
	@Override
	public int compareTo(Scheduling scheduling) {
		int compareResult = 0;
		
		compareResult = cronExpression.compareTo(scheduling.cronExpression);
		if(compareResult!=0){
			return compareResult;
		}
		
		compareResult = delay.compareTo(scheduling.delay);
		if(compareResult!=0){
			return compareResult;
		}
		
		compareResult = (new Integer(schedulingTimes)).
				compareTo((new Integer(scheduling.schedulingTimes)));
		if(compareResult!=0){
			return compareResult;
		}

		compareResult = firstStartTime.compareTo(scheduling.firstStartTime);
		if(compareResult!=0){
			return compareResult;
		}
		
		compareResult = endTime.compareTo(scheduling.endTime);
		if(compareResult!=0){
			return compareResult;
		}
		
		return (new Boolean(previuosExecutionsMustBeCompleted))
				.compareTo(
					new Boolean(scheduling.previuosExecutionsMustBeCompleted));
	}
	
}
