package org.gcube.vremanagement.executor;

import java.lang.reflect.Constructor;
import java.util.Map;
import java.util.UUID;

import javax.jws.WebService;

import org.gcube.vremanagement.executor.api.SmartExecutor;
import org.gcube.vremanagement.executor.api.types.LaunchParameter;
import org.gcube.vremanagement.executor.exception.ExecutorException;
import org.gcube.vremanagement.executor.exception.InputsNullException;
import org.gcube.vremanagement.executor.exception.LaunchException;
import org.gcube.vremanagement.executor.exception.PluginInstanceNotFoundException;
import org.gcube.vremanagement.executor.exception.PluginNotFoundException;
import org.gcube.vremanagement.executor.persistence.JDBCPersistence;
import org.gcube.vremanagement.executor.persistence.JDBCPersistenceConnector;
import org.gcube.vremanagement.executor.persistence.Persistence;
import org.gcube.vremanagement.executor.plugin.Plugin;
import org.gcube.vremanagement.executor.plugin.PluginDeclaration;
import org.gcube.vremanagement.executor.plugin.PluginState;
import org.gcube.vremanagement.executor.pluginmanager.PluginManager;
import org.gcube.vremanagement.executor.pluginmanager.PluginThread;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Effective implementation of Executor
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
@WebService(
portName = "SmartExecutorPort",
serviceName = SmartExecutor.WEB_SERVICE_SERVICE_NAME, 
targetNamespace = SmartExecutor.TARGET_NAMESPACE,
endpointInterface = "org.gcube.vremanagement.executor.api.SmartExecutor" )
public class SmartExecutorImpl implements SmartExecutor {
	
	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(SmartExecutorImpl.class);
	
	/**{@inheritDoc}*/
	@Override
	public String launch(LaunchParameter parameter) throws InputsNullException, 
		PluginNotFoundException, LaunchException, ExecutorException {
		
		Map<String, Object> inputs = parameter.getInputs();
		if(inputs==null){
			throw new InputsNullException();
		}
		
		String name = parameter.getName();
		// Retrieve the PluginDeclaration class representing the plugin which 
		// have the name provided as input 
		logger.debug(String.format("Trying to instatiate a Plugin named %s", name));
		PluginDeclaration pluginDeclaration = PluginManager.getInstance().getPlugin(name);
		if(pluginDeclaration == null){
			throw new PluginNotFoundException();
		}
		
		// Creating the UUID to associate to plugin instance to be run
		UUID executionIdentifier = UUID.randomUUID();
		
		// Retrieving the plugin instance class to be run from PluginDeclaration
		Class<? extends Plugin<? extends PluginDeclaration>> plugin = pluginDeclaration.getPluginImplementation();
		logger.debug(String.format("The class wich will run the execution will be %s", plugin.getName()));
		
		// Retrieve the Constructor of Plugin to instantiate it
		@SuppressWarnings("rawtypes")
		Class[] argTypes = { pluginDeclaration.getClass() , Persistence.class };
		logger.debug(String.format("Plugin named %s once instatiated will be identified by the UUID %s", name, executionIdentifier));
		Constructor<? extends Plugin<? extends PluginDeclaration>> executorPluginConstructor;
		try {
			executorPluginConstructor = plugin.getDeclaredConstructor(argTypes);
		} catch (Exception e) {
			throw new LaunchException();
		} 
		
		JDBCPersistenceConnector jdbcPersistenceConnector = SmartExecutorInitalizator.getJdbcPersistenceConnector();
		// Create and instance of DB connection used to persist plugin evolution
		JDBCPersistence jdbcEvolutionPersistence = new JDBCPersistence(jdbcPersistenceConnector, name, executionIdentifier);
		
		// Create the Argument to pass to contructor
		Object[] arguments = { pluginDeclaration, jdbcEvolutionPersistence};
		
		// Instancing the plugin
		Plugin<? extends PluginDeclaration> instantiatedPlugin;
		try {
			instantiatedPlugin = executorPluginConstructor.newInstance(arguments);
		} catch(Exception e) {
			throw new LaunchException();
		} 
		logger.debug(String.format("Plugin named %s identified by the UUID %s has been instantiated", name, executionIdentifier));
		
		// Creating the thread used to launch the plugin execution
		PluginThread<Plugin<? extends PluginDeclaration>> pluginThread = 
				new PluginThread<Plugin<? extends PluginDeclaration>>(instantiatedPlugin, inputs, executionIdentifier);
		// Adding the thread to the pluginInstances
		SmartExecutorInitalizator.getPluginInstances().put(executionIdentifier, pluginThread);
		
		// Launching Thread from initially created pool 
		SmartExecutorInitalizator.getPool().execute(pluginThread);
		
		logger.debug(String.format("The Plugin named %s with UUID %s has been launched with the provided inputs", name, executionIdentifier));
		
		// TODO join the thread
		
		return executionIdentifier.toString();
	}

	/**{@inheritDoc}*/
	@Override
	public PluginState getState(String executionIdentifier) 
			throws PluginInstanceNotFoundException, ExecutorException {
		try {
			JDBCPersistenceConnector jdbcPersistenceConnector = SmartExecutorInitalizator.getJdbcPersistenceConnector();
			return jdbcPersistenceConnector.getPluginInstanceState(UUID.fromString(executionIdentifier));
		} catch (Exception e) {
			throw new PluginInstanceNotFoundException();
		}
	}


}
