package org.gcube.common.security.secrets;

import java.util.Map;
import org.gcube.common.security.Owner;



/**
 * An abstract class representing a secret used for authorization.
 * <p>
 * A secret encapsulates sensitive information required to authenticate an
 * {@link Owner} and authorize access to resources within a specific context.
 * This class provides a contract for retrieving key security attributes
 * such as the owner, context, and HTTP authorization headers, while also
 * offering methods to check the secret's validity and expiration status.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public abstract class Secret {

	/**
	 * The priority of this secret.
	 * Higher priority secrets are preferred when we try to set multiple secret
	 * in SecretManagerProvider.
	 * @return the priority as an integer.
	 */
	public abstract int priority();


	/**
	 * Retrieves the {@link Owner} associated with this secret.
	 * The owner represents the entity (user or application) that possesses
	 * and uses the secret.
	 *
	 * @return The {@link Owner} of this secret.
	 */
	public abstract Owner getOwner();
	
	/**
	 * Retrieves the authorization context to which this secret belongs.
	 * The context is typically a string representing a specific environment,
	 * service, or scope (e.g., a VO or a VRE).
	 *
	 * @return The string representing the context.
	 */
	public abstract String getContext();
	
	/**
	 * Retrieves a map of HTTP authorization headers to be used with this secret.
	 * These headers can be directly added to an HTTP request for authentication.
	 *
	 * @return A {@code Map<String, String>} containing the header names and their
	 * corresponding values.
	 */
	public abstract Map<String, String> getHTTPAuthorizationHeaders();
	
	/**
	 * Checks if the secret is currently valid and usable for authorization.
	 * A secret may be invalid if it has been revoked or has not yet become active.
	 *
	 * @return {@code true} if the secret is valid, {@code false} otherwise.
	 */
	public abstract boolean isValid();
	
	/**
	 * Checks if the secret has expired.
	 * An expired secret should no longer be used for authorization.
	 *
	 * @return {@code true} if the secret has expired, {@code false} otherwise.
	 */
	public abstract boolean isExpired();
}