package org.gcube.common.iam;

import org.gcube.common.keycloak.KeycloakClient;
import org.gcube.common.keycloak.KeycloakClientException;
import org.gcube.common.keycloak.model.TokenResponse;

/**
 * D4Science IAM client authentication implementation for user credentials.
 * Handles authentication using username and password credentials with optional context.
 * 
 * @author Mauro Mugnaini (Nubisware S.r.l.)
 */
public class D4ScienceIAMClientAuthn4User extends D4ScienceIAMClientAuthn {

    /**
     * Creates a new user authentication instance without context.
     * 
     * @param iamClient the IAM client
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param username the username
     * @param password the password
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected D4ScienceIAMClientAuthn4User(D4ScienceIAMClient iamClient, String clientId, String clientSecret,
            String username, String password) throws D4ScienceIAMClientException {

        this(iamClient, clientId, clientSecret, username, password, null);
    }

    /**
     * Creates a new user authentication instance with context.
     * 
     * @param iamClient the IAM client
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param username the username
     * @param password the password
     * @param context the context (optional)
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected D4ScienceIAMClientAuthn4User(D4ScienceIAMClient iamClient, String clientId, String clientSecret,
            String username, String password, String context) throws D4ScienceIAMClientException {

        super(iamClient, performUserAuthn(iamClient, clientId, clientSecret, username, password, context));
    }

    /**
     * Performs user authentication using username and password credentials.
     * 
     * @param iamClient the IAM client
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param username the username
     * @param password the password
     * @param context the context (optional)
     * @return the token response
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected static final TokenResponse performUserAuthn(D4ScienceIAMClient iamClient, String clientId,
            String clientSecret, String username, String password, String context) throws D4ScienceIAMClientException {

        KeycloakClient keycloakClient = iamClient.getKeycloakClient();
        try {
            return keycloakClient.queryOIDCTokenOfUserWithContext(
                    keycloakClient.getTokenEndpointURL(iamClient.getRealmBaseURL()), clientId, clientSecret, username,
                    password, context);

        } catch (KeycloakClientException e) {
            throw new D4ScienceIAMClientException(e);
        }
    }

}