package org.gcube.common.software.examples;

import java.io.File;
import java.util.List;

import org.gcube.common.software.analyser.Analyser;
import org.gcube.common.software.analyser.AnalyserFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Simple example demonstrating how to use the Software Versions Processor Library
 * 
 * This example shows:
 * 1. Loading configuration from a JSON file
 * 2. Setting up output directory
 * 3. Processing software artifacts
 * 4. Handling the generated output files
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public class SimpleProcessorExample {
    
    private static final Logger logger = LoggerFactory.getLogger(SimpleProcessorExample.class);
    
    public static void main(String[] args) throws Exception {
        
        // Check command line arguments
        if (args.length < 1) {
            System.err.println("Usage: java SimpleProcessorExample <config.json> [output-directory]");
            System.err.println("Example: java SimpleProcessorExample docs/examples/gcat-doc.json ./output");
            System.exit(1);
        }
        
        String configPath = args[0];
        String outputPath = args.length > 1 ? args[1] : "./output";
        
        try {
            // 1. Load configuration from JSON file
            File configFile = new File(configPath);
            if (!configFile.exists()) {
                throw new RuntimeException("Configuration file not found: " + configPath);
            }
            
            logger.info("Loading configuration from: {}", configFile.getAbsolutePath());
            Analyser analyser = AnalyserFactory.getAnalyser(configFile);
            
            // 2. Set output directory
            File outputDirectory = new File(outputPath);
            if (!outputDirectory.exists()) {
                outputDirectory.mkdirs();
                logger.info("Created output directory: {}", outputDirectory.getAbsolutePath());
            }
            analyser.setOutputDirectory(outputDirectory);
            
            // 3. Process all software artifacts
            logger.info("Starting software artifact processing...");
            List<File> outputFiles = analyser.analyse();
            
            // 4. Report results
            logger.info("Processing completed successfully!");
            logger.info("Generated {} output files:", outputFiles.size());
            
            for (File file : outputFiles) {
                logger.info("  - {} ({} bytes)", file.getName(), file.length());
            }
            
            // Show output directory contents
            logger.info("\nOutput directory contents:");
            File[] allFiles = outputDirectory.listFiles();
            if (allFiles != null) {
                for (File file : allFiles) {
                    logger.info("  - {} ({} bytes)", file.getName(), file.length());
                }
            }
            
        } catch (Exception e) {
            logger.error("Failed to process software artifacts", e);
            System.exit(1);
        }
    }
}
