package org.gcube.informationsystem.resourceregistry.publisher;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCacheRenewal;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaViolationException;
import org.gcube.informationsystem.resourceregistry.api.request.BaseRequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath.SharingOperation;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.utils.TypeUtility;
import org.gcube.informationsystem.utils.UUIDManager;
import org.gcube.informationsystem.utils.UUIDUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of the ResourceRegistryPublisher interface.
 * Provides concrete implementation for all Resource Registry operations
 * including CRUD operations for Resources, Facets, and Relations,
 * as well as context management functionality.
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistryPublisherImpl extends BaseRequestInfo implements ResourceRegistryPublisher {

	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryPublisherImpl.class);

	private static final String ACCEPT_HTTP_HEADER_KEY = "Accept";
	private static final String CONTENT_TYPE_HTTP_HEADER_KEY = "Content-Type";
	
	/** The base address of the Resource Registry service */
	protected final String address;
	
	/** HTTP headers to be included in requests */
	protected Map<String, String> headers;
	
	/** Cache for context information to improve performance */
	protected ContextCache contextCache;
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isHierarchicalMode() {
		return hierarchicalMode;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHierarchicalMode(boolean hierarchicalMode) {
		this.hierarchicalMode = hierarchicalMode;
	}
	
	private void addOptionalQueryParameters(Map<String,String> queryParams) throws UnsupportedEncodingException {
		addHierarchicalMode(queryParams);
		addIncludeContexts(queryParams);
		addIncludeMeta(queryParams);
		addIncludeAllMeta(queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		Map<String,String> queryParams = new HashMap<>();
		return includeAdditionalQueryParameters(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(queryParams==null) {
			queryParams = new HashMap<>();
		}
		addOptionalQueryParameters(queryParams);
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private void addHierarchicalMode(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(hierarchicalMode) {
			queryParams.put(InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER, Boolean.toString(hierarchicalMode));
		}
	}
	
	private void addIncludeContexts(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(includeContexts) {
			queryParams.put(InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER, Boolean.toString(includeContexts));
		}
	}
	
	private void addIncludeMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addIncludeMeta(queryParams, includeMeta);
	}
	
	private void addIncludeMeta(Map<String,String> queryParams, boolean includeMeta) throws UnsupportedEncodingException{
		if(includeMeta) {
			queryParams.put(InstancePath.INCLUDE_META_QUERY_PARAMETER, Boolean.toString(includeMeta));
		}
	}
	
	private void addIncludeAllMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(allMeta) {
			queryParams.put(InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER, Boolean.toString(allMeta));
		}
	}
	
	private void addOffset(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addOffset(queryParams, offset);
	}
	
	private void addOffset(Map<String,String> queryParams, Integer offset) throws UnsupportedEncodingException{
		if(offset!=null) {
			queryParams.put(InstancePath.OFFSET_QUERY_PARAMETER, offset.toString());
		}
	}
	
	private void addLimit(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addLimit(queryParams, limit);
	}
	
	private void addLimit(Map<String,String> queryParams, Integer limit) throws UnsupportedEncodingException{
		if(limit!=null) {
			queryParams.put(InstancePath.LIMIT_QUERY_PARAMETER, limit.toString());
		}
	}

	/** Context cache renewal strategy for refreshing cached context information */
	protected ContextCacheRenewal contextCacheRenewal = new ContextCacheRenewal() {
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public List<Context> renew() throws ResourceRegistryException {
			return getAllContextFromServer(true, 0, BaseRequestInfo.UNBOUNDED_LIMIT);
		}
		
	};
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, String value) {
		headers.put(name, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, boolean value) {
		addHeader(name, Boolean.toString(value));
	}
	
	/**
	 * Creates and configures a new HTTP request with authorization headers and custom headers.
	 * 
	 * @return A configured GXHTTPStringRequest instance ready for use
	 */
	protected GXHTTPStringRequest getGXHTTPStringRequest() {
		GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
		
		/* It is a gcube request */
		Secret secret = SecretManagerProvider.get();
		Map<String, String> authorizationHeaders = secret.getHTTPAuthorizationHeaders();
		for(String key : authorizationHeaders.keySet()) {
			gxHTTPStringRequest.header(key, authorizationHeaders.get(key));
		}
		
		gxHTTPStringRequest.from(this.getClass().getSimpleName());
		for(String name : headers.keySet()) {
			gxHTTPStringRequest.header(name, headers.get(name));
		}
		return gxHTTPStringRequest;
	}

	/**
	 * Creates a new ResourceRegistryPublisherImpl instance with a shared context cache.
	 * 
	 * @param address The base address of the Resource Registry service
	 */
	public ResourceRegistryPublisherImpl(String address) {
		this(address, true);
	}
	
	/**
	 * Creates a new ResourceRegistryPublisherImpl instance.
	 * 
	 * @param address The base address of the Resource Registry service
	 * @param sharedContextCache Whether to use a shared context cache instance or create a new one
	 */
	public ResourceRegistryPublisherImpl(String address, boolean sharedContextCache) {
		super();
		this.address = address;
		this.headers = new HashMap<>();
		if(sharedContextCache) {
			contextCache = ContextCache.getInstance();
		}else {
			contextCache = new ContextCache();
		}
		contextCache.setContextCacheRenewal(contextCacheRenewal);
	}

	/**
	 * It reads all the contexts from server. 
	 * The cache used for contexts is bypassed and not updated.
	 * @return All Contexts read from server
	 * @throws ResourceRegistryException if there is an error communicating with the server
	 */
	public List<Context> getAllContextFromServer() throws ResourceRegistryException {
		return getAllContextFromServer(includeMeta, offset, limit);
	}
	
	/**
	 * Retrieves all contexts from the Resource Registry server.
	 * 
	 * @param includeMeta Whether to include metadata in the response
	 * @param offset The offset for pagination
	 * @param limit The limit for pagination
	 * @return List of all contexts from the server
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	protected List<Context> getAllContextFromServer(boolean includeMeta, Integer offset, Integer limit) throws ResourceRegistryException {
		try {
			logger.info("Going to read all {}s", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters, includeMeta);
			addOffset(parameters, offset);
			addLimit(parameters, limit);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.debug("Got Contexts are {}", ret);
			return ElementMapper.unmarshalList(Context.class, ret);

		} catch (ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Context> getAllContexts() throws ResourceRegistryException {
		return contextCache.getContexts();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ContextCache getContextCache() {
		return contextCache;
	}

	/**
	 * It reads the context from server. 
	 * The cache used for contexts is bypassed and not updated.
	 * @param uuid the UUID of the context to read
	 * @return the Contexts read from server
	 * @throws ContextNotFoundException if the context is not found
	 * @throws ResourceRegistryException if there is an error communicating with the server
	 */
	public Context getContextFromServer(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		return getContextFromServer(uuid.toString());
	}
	
	/**
	 * Retrieves a specific context from the Resource Registry server.
	 * 
	 * @param uuid The UUID of the context to retrieve
	 * @return The context with the specified UUID
	 * @throws ContextNotFoundException If the context with the specified UUID is not found
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	protected Context getContextFromServer(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.info("Going to get current {} ", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid);

			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			Context context = HTTPUtility.getResponse(Context.class, httpURLConnection);

			logger.debug("Got Context is {}", ElementMapper.marshal(context));
			return context;
		} catch (ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		Context context = contextCache.getContextByUUID(uuid);
		if (context == null) {
			context = getContextFromServer(uuid.toString());
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(uuid);
			if (c != null) {
				context = c;
			} else {
				logger.error(
						"Context with UUID {} is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.",
						uuid, context);
			}
		}
		return context;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException {
		String contextFullName = SecretManagerProvider.get().getContext();
		UUID uuid = contextCache.getUUIDByFullName(contextFullName);
		return getContext(uuid);
	}

	private UUID getCurrentContextUUID() throws ResourceRegistryException {
		return getCurrentContext().getID();
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Type Resolution:</strong>
	 * <ul>
	 * <li>Uses {@link TypeUtility#getTypeName(Class)} to derive the type name from the class</li>
	 * <li>Delegates to {@link #list(String, Boolean)} for the actual REST API call</li>
	 * <li>Unmarshals the JSON response back to typed objects using {@link ElementMapper#unmarshalList(String)}.</li>
	 * </ul>
	 * 
	 * <strong>Request Configuration:</strong>
	 * <ul>
	 * <li>Builds request to {@code GET /instances/{type-name}} endpoint</li>
	 * <li>Automatically includes offset and limit parameters from base configuration</li>
	 * <li>Sets polymorphic query parameter based on the provided boolean value</li>
	 * <li>Uses application/json content type for the response.</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Receives JSON array from server</li>
	 * <li>Unmarshals JSON to List of typed objects using {@link ElementMapper}</li>
	 * <li>Returns strongly-typed list for type-safe usage.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s propagated directly from server</li>
	 * <li>JSON marshalling/unmarshalling errors are wrapped in {@link RuntimeException};.</li>
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> List<ERElem> list(Class<ERElem> clazz, Boolean polymorphic)
			throws ResourceRegistryException {
		String type = TypeUtility.getTypeName(clazz);
		String ret = list(type, polymorphic);
		try {
			return (List<ERElem>) ElementMapper.unmarshalList(ERElement.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Call:</strong>
	 * <ul>
	 * <li>Makes HTTP GET request to {@code /instances/{type-name}} endpoint</li>
	 * <li>Sets Accept header to application/json for JSON response</li>
	 * <li>Includes type name as path parameter.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>Automatically includes offset and limit from base configuration (if set)</li>
	 * <li>Adds polymorphic parameter based on the provided boolean value</li>
	 * <li>Additional query parameters from base class configuration are included.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Returns raw JSON string response from server</li>
	 * <li>Response contains JSON array of instances matching the criteria</li>
	 * <li>Server filters results based on current context and user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Configuration:</strong>
	 * <ul>
	 * <li>Uses {@link GXHTTPStringRequest} for HTTP communication</li>
	 * <li>Includes authorization headers automatically via base request configuration</li>
	 * <li>Response processed via {@link HTTPUtility#getResponse(Class, HttpURLConnection)} for proper error handling.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s propagated directly from server</li>
	 * <li>Other exceptions wrapped in {@link RuntimeException}</li>
	 * <li>HTTP errors converted to appropriate Resource Registry exceptions.</li>
	 * </ul>
	 */
	@Override
	public String list(String type, Boolean polymorphic) throws ResourceRegistryException {
		try {
			logger.info("Going to get all instances of {} ", type);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			
			Map<String,String> parameters = new HashMap<>();
			addOffset(parameters);
			addLimit(parameters);
			parameters.put(InstancePath.POLYMORPHIC_QUERY_PARAMETER, polymorphic.toString());
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got instances of {} are {}", type, ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * Creates a new instance of the specified type in the Resource Registry.
	 * 
	 * @param type The type of the instance to create
	 * @param json The JSON representation of the instance
	 * @param uuid The UUID for the new instance
	 * @return The JSON representation of the created instance
	 * @throws SchemaViolationException If the instance violates schema constraints
	 * @throws AlreadyPresentException If an instance with the same UUID already exists
	 * @throws ResourceRegistryException If an error occurs during creation
	 */
	protected String create(String type, String json, UUID uuid)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", type, json);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());

			Map<String,String> parameters = new HashMap<>();
			addIncludeContexts(parameters);
			addIncludeMeta(parameters);
			addIncludeAllMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);
			
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.trace("{} successfully created", ret);
			return ret;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * Internal method to create an Information System element.
	 * Handles UUID generation if not present and marshals the element to JSON.
	 * 
	 * @param <ERElem> The type of Information System element
	 * @param er The element to create
	 * @return The JSON representation of the created element
	 * @throws SchemaViolationException If the element violates schema constraints
	 * @throws AlreadyPresentException If an element with the same UUID already exists
	 * @throws ResourceRegistryException If an error occurs during creation
	 */
	protected <ERElem extends ERElement> String internalCreate(ERElem er)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			UUID uuid = er.getID();
			if (uuid == null) {
				uuid = UUIDManager.getInstance().generateValidUUID();
				er.setID(uuid);
			}
			String json = ElementMapper.marshal(er);
			return create(type, json, uuid);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Generation:</strong>
	 * <ul>
	 * <li>If the instance doesn't have a UUID set, generates a new valid UUID using {@link UUIDManager}</li>
	 * <li>Sets the generated UUID on the instance before marshalling to JSON.</li>
	 * </ul>
	 * 
	 * <strong>Marshalling Process:</strong>
	 * <ul>
	 * <li>Uses {@link ElementMapper#marshal(Element)} to convert the instance to JSON representation</li>
	 * <li>Type name is extracted from the instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)} method</li>
	 * <li>Delegates to the internal {@link #create(String, String, UUID)} method for the actual REST call.</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PUT request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept and Content-Type headers to application/json with UTF-8 charset</li>
	 * <li>Includes query parameters for metadata and context information based on client configuration.</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Receives JSON response from server containing the created instance with metadata</li>
	 * <li>Unmarshals the JSON response back to the original object type using {@link ElementMapper}</li>
	 * <li>Returns the typed instance with server-generated properties (e.g., creation timestamp, metadata).</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>JSON marshalling/unmarshalling errors are wrapped in {@link RuntimeException}</li>
	 * <li>HTTP communication errors are handled by the underlying HTTP utility.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters Included:</strong>
	 * <ul>
	 * <li>{@code includeContexts}: Whether to include context information in the response</li>
	 * <li>{@code includeMeta}: Whether to include metadata in the response</li>
	 * <li>{@code allMeta}: Whether to include metadata for all nested instances.</li>
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> ERElem create(ERElem er)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException {
		try {
			String ret = internalCreate(er);
			return (ERElem) ElementMapper.unmarshal(ERElement.class, ret);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>JSON Processing:</strong>
	 * <ul>
	 * <li>Uses Jackson {@link ObjectMapper} to parse the input JSON string into a {@link JsonNode}</li>
	 * <li>Extracts the type name from the JSON using {@link TypeUtility#getTypeName(JsonNode)}</li>
	 * <li>Retrieves or generates UUID using {@link UUIDUtility#getUUID(JsonNode)}.</li>
	 * </ul>
	 * 
	 * <strong>UUID Handling:</strong>
	 * <ul>
	 * <li>If no UUID is present in the JSON, generates a new valid UUID using {@link UUIDManager}</li>
	 * <li>Adds the generated UUID to the JSON structure</li>
	 * <li>Serializes the updated JSON back to string format for the REST call.</li>
	 * </ul>
	 * 
	 * <strong>Type Resolution:</strong>
	 * <ul>
	 * <li>Type name must be present in the JSON as @class property</li>
	 * <li>Uses {@link TypeUtility} to extract the type name from the JSON structure</li>
	 * <li>Type name is used as path parameter in the REST API call.</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PUT request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept and Content-Type headers to application/json with UTF-8 charset</li>
	 * <li>Sends the JSON string as request body</li>
	 * <li>Includes query parameters for metadata and context information.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Returns the raw JSON string response from the server</li>
	 * <li>Response contains the created instance with server-generated metadata and properties</li>
	 * <li>No unmarshalling is performed, allowing callers to handle JSON as needed.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>JSON parsing errors are wrapped in {@link RuntimeException}</li>
	 * <li>Missing @class property or invalid JSON structure cause exceptions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters Included:</strong>
	 * <ul>
	 * <li>{@code includeContexts}: Whether to include context information in the response</li>
	 * <li>{@code includeMeta}: Whether to include metadata in the response</li>
	 * <li>{@code allMeta}: Whether to include metadata for all nested instances.</li>
	 * </ul>
	 */
	@Override
	public String create(String json)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException {
		try {
			ObjectMapper mapper = new ObjectMapper();
			JsonNode jsonNode = mapper.readTree(json);
			String type = TypeUtility.getTypeName(jsonNode);
			UUID uuid = UUIDUtility.getUUID(jsonNode);
			if(uuid==null) {
				uuid = UUIDManager.getInstance().generateValidUUID();
				((ObjectNode) jsonNode).put(ERElement.ID_PROPERTY, uuid.toString());
				json = mapper.writeValueAsString(jsonNode);
			}
			return create(type, json, uuid);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Type and UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts the type name from the instance using {@link ERElement#getTypeName()}</li>
	 * <li>Retrieves the UUID from the instance using {@link ERElement#getID()}</li>
	 * <li>Delegates to {@link #exist(String, UUID)} for the actual implementation.</li>
	 * </ul>
	 * 
	 * <strong>Validation:</strong>
	 * <ul>
	 * <li>The instance must have a valid UUID set, otherwise the check will fail</li>
	 * <li>The instance's type must be a valid Information System model type.</li>
	 * </ul>
	 */
	@Override
	public <ERElem extends ERElement> boolean exist(ERElem er)
			throws AvailableInAnotherContextException, ResourceRegistryException {
		String type = er.getTypeName();
		UUID uuid = er.getID();
		return exist(type, uuid);
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Type Resolution:</strong>
	 * <ul>
	 * <li>Derives the type name from the provided class using {@link TypeUtility#getTypeName(Class)}</li>
	 * <li>Ensures type-safe operation by resolving the exact type name from the class</li>
	 * <li>Delegates to {@link #exist(String, UUID)} for the actual implementation.</li>
	 * </ul>
	 * 
	 * <strong>Class Validation:</strong>
	 * <ul>
	 * <li>The class must be a valid Information System model type</li>
	 * <li>Type name resolution is performed using the TypeUtility for consistency.</li>
	 * </ul>
	 */
	@Override
	public <ERElem extends ERElement> boolean exist(Class<ERElem> clazz, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException {
		String type = TypeUtility.getTypeName(clazz);
		return exist(type, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Request Configuration:</strong>
	 * <ul>
	 * <li>Makes HTTP HEAD request to the instances endpoint</li>
	 * <li>Includes hierarchical mode parameter if enabled via client configuration.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Successful response indicates the instance exists and is accessible</li>
	 * <li>NotFoundException is caught and converted to false return value</li>
	 * <li>Other exceptions are propagated according to the interface contract.</li>
	 * </ul>
	 * 
	 * <strong>Authentication:</strong>
	 * <ul>
	 * <li>Uses automatic authorization headers via the base request configuration</li>
	 * <li>User permissions are validated server-side.</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Logs info-level message when starting the existence check</li>
	 * <li>Logs debug-level message when instance is confirmed to exist</li>
	 * <li>Uses type and UUID in log messages for traceability.</li>
	 * </ul>
	 */
	@Override
	public boolean exist(String type, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to check if {} with UUID {} exists", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());

			Map<String,String> queryParams =  new HashMap<>();
			addHierarchicalMode(queryParams);
			gxHTTPStringRequest.queryParams(queryParams);

			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.debug("{} with UUID {} exists", type, uuid);
			return true;
		} catch (NotFoundException e) {
			return false;
		} catch (ResourceRegistryException e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Type and UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts the type name from the instance using {@link ERElement#getTypeName()}</li>
	 * <li>Retrieves the UUID from the instance using {@link ERElement#getID()}</li>
	 * <li>Delegates to {@link #read(String, UUID)} for the actual REST API call.</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Receives JSON response from the delegated method</li>
	 * <li>Unmarshals the JSON back to the original object type using {@link ElementMapper#unmarshal(Class, String)}</li>
	 * <li>Performs type casting to return the strongly-typed instance</li>
	 * <li>Returns the complete instance with all properties and server metadata.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>Uses @SuppressWarnings("unchecked") for the type casting from {@link ERElement} to {@link ERElem}</li>
	 * <li>Type safety is ensured by the fact that the JSON contains the same type as the input instance</li>
	 * <li>The returned object will be of the same type as the provided instance.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from the delegated method are propagated directly</li>
	 * <li>JSON unmarshalling errors are wrapped in {@link RuntimeException}</li>
	 * <li>Server-side errors (404, 403, etc.) are handled by the underlying {@link #read(String, UUID)} method.</li>
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> ERElem read(ERElem er)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			UUID uuid = er.getID();
			String ret = read(type, uuid);
			return (ERElem) ElementMapper.unmarshal(ERElement.class, ret);
		} catch (ResourceRegistryException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Type Resolution:</strong>
	 * <ul>
	 * <li>Derives the type name from the provided class using {@link TypeUtility#getTypeName(Class)}</li>
	 * <li>Ensures type-safe operation by resolving the exact type name from the class</li>
	 * <li>Delegates to {@link #read(String, UUID)} for the actual REST API call.</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Receives JSON response from the delegated method</li>
	 * <li>Unmarshals the JSON to the specified class type using {@link ElementMapper#unmarshal(Class, String)}</li>
	 * <li>Performs type casting to return the strongly-typed instance</li>
	 * <li>Returns the complete instance with all properties and server metadata.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>Uses @SuppressWarnings("unchecked") for the type casting from {@link ERElement} to {@link ERElem}</li>
	 * <li>Type safety is ensured by using the class parameter for type resolution</li>
	 * <li>The returned object will be of exactly the specified class type.</li>
	 * </ul>
	 * 
	 * <strong>Class Validation:</strong>
	 * <ul>
	 * <li>The class must be a valid Information System model type</li>
	 * <li>Type name resolution is performed using the TypeUtility for consistency with other methods.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from the delegated method are propagated directly</li>
	 * <li>JSON unmarshalling errors are wrapped in {@link RuntimeException}.</li>
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> ERElem read(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			String type = TypeUtility.getTypeName(clazz);
			String ret = read(type, uuid);
			return (ERElem) ElementMapper.unmarshal(ERElement.class, ret);
		} catch (ResourceRegistryException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Call:</strong>
	 * <ul>
	 * <li>Makes HTTP GET request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept header to application/json with UTF-8 charset for JSON response</li>
	 * <li>Includes type name and UUID as path parameters.</li>
	 * </ul>
	 * 
	 * <strong>Request Configuration:</strong>
	 * <ul>
	 * <li>Uses {@link GXHTTPStringRequest} for HTTP communication</li>
	 * <li>Includes authorization headers automatically via base request configuration</li>
	 * <li>Automatically includes additional query parameters from client configuration:</li>
	 *   <li>{@code hierarchicalMode}: Whether to search in parent contexts</li>
	 *   <li>{@code includeContexts}: Whether to include context information in response</li>
	 *   <li>{@code includeMeta}: Whether to include metadata in response</li>
	 *   <li>{@code allMeta}: Whether to include metadata for all nested instances.</li>
	 *   </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>Additional query parameters are included</li>
	 * <li>Parameters are based on the current client configuration state</li>
	 * <li>Supports hierarchical context navigation when enabled.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Returns raw JSON string response from server</li>
	 * <li>Response contains complete instance with all properties and metadata</li>
	 * <li>Server filters results based on current context and user authorization</li>
	 * <li>Response processed via {@link HTTPUtility#getResponse(Class, HttpURLConnection)} for proper error handling.</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Logs trace-level message when starting the read operation with type and UUID</li>
	 * <li>Logs debug-level message with the complete JSON response when successful</li>
	 * <li>Uses structured logging for better traceability and debugging.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s propagated directly from server</li>
	 * <li>Other exceptions wrapped in {@link RuntimeException}.</li>
	 * </ul>
	 */
	@Override
	public String read(String type, UUID uuid) 
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());

			includeAdditionalQueryParameters(gxHTTPStringRequest);

			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.debug("Got {} with UUID {} is {}", type, uuid, ret);
			return ret;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Instance Validation:</strong>
	 * <ul>
	 * <li>The instance must have a valid UUID set, as updates require identification of the existing instance</li>
	 * <li>The instance's type must match an existing instance in the Resource Registry.</li>
	 * </ul>
	 * 
	 * <strong>Marshalling Process:</strong>
	 * <ul>
	 * <li>Uses {@link ElementMapper#marshal(Element)} to convert the instance to JSON representation</li>
	 * <li>Type name is extracted from the instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)} method</li>
	 * <li>UUID is extracted from the instance using {@link org.gcube.informationsystem.base.reference.IdentifiableElement#getID()} method</li>
	 * <li>Delegates to the internal {@link #update(String, String, UUID)} method for the actual REST call.</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PUT request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept and Content-Type headers to application/json with UTF-8 charset</li>
	 * <li>Includes query parameters for metadata and context information based on client configuration.</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Receives JSON response from server containing the updated instance with metadata</li>
	 * <li>Unmarshals the JSON response back to the original object type using {@link ElementMapper}</li>
	 * <li>Returns the typed instance with server-updated properties (e.g., modification timestamp, metadata).</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>JSON marshalling/unmarshalling errors are wrapped in {@link RuntimeException}</li>
	 * <li>HTTP communication errors are handled by the underlying HTTP utility.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters Included:</strong>
	 * <ul>
	 * <li>{@code includeContexts}: Whether to include context information in the response</li>
	 * <li>{@code includeMeta}: Whether to include metadata in the response</li>
	 * <li>{@code allMeta}: Whether to include metadata for all nested instances.</li>
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> ERElem update(ERElem er)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			String json = ElementMapper.marshal(er);
			UUID uuid = er.getID();
			String ret = update(type, json, uuid);
			return (ERElem) ElementMapper.unmarshal(ERElement.class, ret);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>JSON Processing:</strong>
	 * <ul>
	 * <li>Uses Jackson {@link ObjectMapper} to parse the input JSON string into a {@link JsonNode}</li>
	 * <li>Extracts the type name from the JSON using {@link TypeUtility#getTypeName(JsonNode)}</li>
	 * <li>The JSON must contain the @class property indicating the instance type.</li>
	 * </ul>
	 * 
	 * <strong>UUID Validation:</strong>
	 * <ul>
	 * <li>The JSON must contain a valid UUID for the instance to be updated</li>
	 * <li>UUID is used to identify the existing instance in the Resource Registry</li>
	 * <li>Delegates to {@link #update(String type, String json)} for further processing.</li>
	 * </ul>
	 * 
	 * <strong>Type Resolution:</strong>
	 * <ul>
	 * <li>Type name must be present in the JSON as @class property</li>
	 * <li>Uses {@link TypeUtility} to extract the type name from the JSON structure</li>
	 * <li>Type name is used as path parameter in the REST API call.</li>
	 * </ul>
	 * 
	 * <strong>Processing Flow:</strong>
	 * <ul>
	 * <li>Extracts type name from JSON structure</li>
	 * <li>Delegates to {@link #update(String, String)} method which handles UUID extraction and REST call</li>
	 * <li>Returns the raw JSON string response from the server.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>JSON parsing errors are wrapped in {@link RuntimeException}</li>
	 * <li>Missing @class property or invalid JSON structure cause exceptions.</li>
	 * </ul>
	 */
	@Override
	public String update(String json)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			String type = TypeUtility.getTypeName(json);
			return update(type, json);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Uses {@link UUIDUtility#getUUID(String)} to extract the UUID from the JSON</li>
	 * <li>The JSON must contain a valid UUID field for instance identification</li>
	 * <li>Delegates to {@link #update(String, String, UUID)} for the actual REST call.</li>
	 * </ul>
	 * 
	 * <strong>JSON Validation:</strong>
	 * <ul>
	 * <li>The JSON must be well-formed and contain all required fields for the update</li>
	 * <li>Type parameter ensures type-safe operation by providing explicit type information</li>
	 * <li>UUID must match an existing instance in the Resource Registry.</li>
	 * </ul>
	 * 
	 * <strong>Processing Flow:</strong>
	 * <ul>
	 * <li>Extracts UUID from the provided JSON</li>
	 * <li>Delegates to the final update method with type, JSON, and UUID parameters</li>
	 * <li>Returns the raw JSON response from the server.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>JSON parsing errors are wrapped in {@link RuntimeException}</li>
	 * <li>Missing or invalid UUID in JSON causes exceptions.</li>
	 * </ul>
	 */
	@Override
	public String update(String type, String json)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			UUID uuid = UUIDUtility.getUUID(json);
			return update(type, json, uuid);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PUT request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept and Content-Type headers to application/json with UTF-8 charset</li>
	 * <li>Includes type name and UUID as path parameters</li>
	 * <li>Sends the JSON string as request body.</li>
	 * </ul>
	 * 
	 * <strong>Request Configuration:</strong>
	 * <ul>
	 * <li>Uses {@link GXHTTPStringRequest} for HTTP communication</li>
	 * <li>Includes authorization headers automatically via base request configuration</li>
	 * <li>Automatically includes additional query parameters from client configuration:</li>
	 *   <li>{@code includeContexts}: Whether to include context information in response</li>
	 *   <li>{@code includeMeta}: Whether to include metadata in response</li>
	 *   <li>{@code allMeta}: Whether to include metadata for all nested instances.</li>
	 *   </ul>
	 * 
	 * <strong>Update Semantics:</strong>
	 * <ul>
	 * <li>Performs a complete replacement of the existing instance with the provided JSON</li>
	 * <li>The UUID must match an existing instance in the current context</li>
	 * <li>All relationships and references are preserved unless explicitly modified in the JSON.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Returns raw JSON string response from server</li>
	 * <li>Response contains the updated instance with all properties and server metadata</li>
	 * <li>Server updates modification timestamps and other metadata automatically</li>
	 * <li>Response processed via {@link HTTPUtility#getResponse(Class, HttpURLConnection)} for proper error handling.</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Logs trace-level message when starting the update operation with type and JSON</li>
	 * <li>Logs trace-level message with the UUID when the update is successful</li>
	 * <li>Uses structured logging for better traceability and debugging.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s propagated directly from server</li>
	 * <li>HTTP 404 errors converted to {@link NotFoundException} if instance doesn't exist</li>
	 * <li>HTTP 403 errors and context accessibility issues converted to appropriate exceptions</li>
	 * <li>Other exceptions wrapped in {@link RuntimeException}.</li>
	 * </ul>
	 */
	@Override
	public String update(String type, String json, UUID uuid)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", type, json);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeContexts(parameters);
			addIncludeMeta(parameters);
			addIncludeAllMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);

			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.trace("{} with UUID {} successfully created : {}", type, uuid, ret);
			return ret;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Instance Identification:</strong>
	 * <ul>
	 * <li>Extracts the type name from the instance using {@link ERElement#getTypeName()}</li>
	 * <li>Retrieves the UUID from the instance using {@link ERElement#getID()}</li>
	 * <li>Delegates to {@link #delete(String, UUID)} for the actual REST API call.</li>
	 * </ul>
	 * 
	 * <strong>Type and UUID Validation:</strong>
	 * <ul>
	 * <li>The instance must have a valid UUID set for identification of the target instance</li>
	 * <li>The instance's type must match an existing instance in the Resource Registry</li>
	 * <li>Both type and UUID are required for the deletion operation.</li>
	 * </ul>
	 * 
	 * <strong>Cascading Effects:</strong>
	 * <ul>
	 * <li>For Resources: Deletion automatically removes all associated ConsistsOf relations</li>
	 * <li>For Facets: Deletion is only allowed if no ConsistsOf relations reference the Facet</li>
	 * <li>For Relations: Deletion removes the relation without affecting source or target instances.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s from server are propagated directly</li>
	 * <li>Missing UUID or invalid instance structure cause exceptions</li>
	 * <li>Referential integrity violations result in appropriate server exceptions.</li>
	 * </ul>
	 */
	@Override
	public <ERElem extends ERElement> boolean delete(ERElem er)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			UUID uuid = er.getID();
			return delete(type, uuid);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Call:</strong>
	 * <ul>
	 * <li>Makes HTTP DELETE request to {@code /instances/{type-name}/{uuid}} endpoint</li>
	 * <li>Sets Accept header to application/json for consistent communication</li>
	 * <li>Includes type name and UUID as path parameters.</li>
	 * </ul>
	 * 
	 * <strong>Request Configuration:</strong>
	 * <ul>
	 * <li>Uses {@link GXHTTPStringRequest} for HTTP communication</li>
	 * <li>Includes authorization headers automatically via base request configuration</li>
	 * <li>No query parameters required for basic deletion operation</li>
	 * <li>No request body is sent with DELETE requests.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Semantics:</strong>
	 * <ul>
	 * <li>Performs permanent removal of the instance from the Resource Registry</li>
	 * <li>The instance must exist in the current context to be deleted</li>
	 * <li>Referential integrity constraints are enforced by the server</li>
	 * <li>Cascading deletion rules apply based on the instance type (Resource, Facet, or Relation).</li>
	 * </ul>
	 * 
	 * <strong>Cascading Behavior:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: All associated ConsistsOf relations are automatically deleted</li>
	 * <li><strong>Facets</strong>: Deletion fails if any ConsistsOf relations reference the Facet</li>
	 * <li><strong>Relations</strong>: Only the relation itself is deleted, source and target instances remain.</li>
	 * </ul>
	 * 
	 * <strong>Response Handling:</strong>
	 * <ul>
	 * <li>Server response is processed but content is not returned to caller</li>
	 * <li>Always returns {@code true} when deletion is successful</li>
	 * <li>Exceptions are thrown for error conditions rather than returning {@code false}</li>
	 * <li>Response processed via {@link HTTPUtility#getResponse(Class, HttpURLConnection)} for proper error handling.</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Logs trace-level message when starting the delete operation with type and UUID</li>
	 * <li>Logs info-level message indicating successful or failed deletion</li>
	 * <li>Uses structured logging for better traceability and debugging.</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException}s propagated directly from server</li>
	 * <li>HTTP 404 errors converted to {@link NotFoundException} if instance doesn't exist</li>
	 * <li>HTTP 403 errors for insufficient permissions or context restrictions</li>
	 * <li>Referential integrity violations result in {@link SchemaViolationException}</li>
	 * <li>Other exceptions wrapped in {@link RuntimeException}.</li>
	 * </ul>
	 */
	@Override
	public boolean delete(String type, UUID uuid)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to delete {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());

			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			HTTPUtility.getResponse(String.class, httpURLConnection);

			boolean deleted = true;

			logger.info("{} with UUID {} {}", type, uuid, deleted ? " successfully deleted" : "was NOT deleted");
			return deleted;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(ERElem)} for implementation details.
	 */
	@Override
	public <F extends Facet> F createFacet(F facet)
			throws SchemaViolationException, FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(String)} for implementation details.
	 */
	@Override
	public String createFacet(String facet)
			throws SchemaViolationException, FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(ERElem)} for implementation details.
	 */
	@Override
	public <F extends Facet> F readFacet(F facet) 
			throws FacetNotFoundException, FacetAvailableInAnotherContextException, ResourceRegistryException {
		return read(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(String, UUID)} for implementation details.
	 */
	@Override
	public String readFacet(String facetType, UUID uuid) 
			throws FacetNotFoundException, FacetAvailableInAnotherContextException, ResourceRegistryException {
		return read(facetType, uuid);
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(ERElem)} for implementation details.
	 */
	@Override
	public <F extends Facet> F updateFacet(F facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(String)} for implementation details.
	 */
	@Override
	public String updateFacet(String facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(ERElem)} for implementation details.
	 */
	@Override
	public <F extends Facet> boolean deleteFacet(F facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException {
		return delete(facet);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(String, UUID)} for implementation details.
	 */
	@Override
	public boolean deleteFacet(String facetType, UUID uuid)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException {
		return delete(facetType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(ERElem)} for implementation details.
	 */
	@Override
	public <R extends Resource> R createResource(R resource)
			throws SchemaViolationException, ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(String)} for implementation details.
	 */
	@Override
	public String createResource(String resource)
			throws SchemaViolationException, ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(ERElem)} for implementation details.
	 */
	@Override
	public <R extends Resource> R readResource(R resource) 
			throws ResourceNotFoundException, ResourceAvailableInAnotherContextException, ResourceRegistryException {
		return read(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(String, UUID)} for implementation details.
	 */
	@Override
	public String readResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceAvailableInAnotherContextException, ResourceRegistryException {
		return read(resourceType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(ERElem)} for implementation details.
	 */
	@Override
	public <R extends Resource> R updateResource(R resource)
			throws SchemaViolationException, ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(String)} for implementation details.
	 */
	@Override
	public String updateResource(String resource)
			throws SchemaViolationException, ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(ERElem)} for implementation details.
	 */
	@Override
	public <R extends Resource> boolean deleteResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resource);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(String, UUID)} for implementation details.
	 */
	@Override
	public boolean deleteResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resourceType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(ERElem)} for implementation details.
	 */
	@Override
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C createConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(String)} for implementation details.
	 */
	@Override
	public String createConsistsOf(String consistsOf)
			throws SchemaViolationException, ConsistsOfAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(ERElem)} for implementation details.
	 */
	@Override
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C readConsistsOf(C consistsOf)
			throws ConsistsOfNotFoundException, ConsistsOfAvailableInAnotherContextException, ResourceRegistryException {
		return read(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(String, UUID)} for implementation details.
	 */
	@Override
	public String readConsistsOf(String consistsOfType, UUID uuid) 
			throws ConsistsOfNotFoundException, ConsistsOfAvailableInAnotherContextException, ResourceRegistryException {
		return read(consistsOfType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(ERElem)} for implementation details.
	 */
	@Override
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C updateConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(String)} for implementation details.
	 */
	@Override
	public String updateConsistsOf(String consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(ERElem)} for implementation details.
	 */
	@Override
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> boolean deleteConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException {
		return delete(consistsOf);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(String, UUID)} for implementation details.
	 */
	@Override
	public boolean deleteConsistsOf(String consistsOfType, UUID uuid)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException {
		return delete(consistsOfType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(ERElem)} for implementation details.
	 */
	@Override
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I createIsRelatedTo(I isRelatedTo)
			throws SchemaViolationException, IsRelatedToAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(String)} for implementation details.
	 */
	@Override
	public String createIsRelatedTo(String isRelatedTo)
			throws SchemaViolationException, IsRelatedToAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(ERElem)} for implementation details.
	 */
	@Override
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I readIsRelatedTo(I isRelatedTo)
			throws IsRelatedToNotFoundException, IsRelatedToAvailableInAnotherContextException, ResourceRegistryException {
		return read(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #read(String, UUID)} for implementation details.
	 */
	@Override
	public String readIsRelatedTo(String isRelatedToType, UUID uuid)
			throws IsRelatedToNotFoundException, IsRelatedToAvailableInAnotherContextException, ResourceRegistryException {
		return read(isRelatedToType, uuid);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(ERElem)} for implementation details.
	 */
	@Override
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I updateIsRelatedTo(I isRelatedTo)
			throws SchemaViolationException, IsRelatedToNotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #update(String)} for implementation details.
	 */
	@Override
	public String updateIsRelatedTo(String isRelatedTo)
			throws SchemaViolationException, IsRelatedToNotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(ERElem)} for implementation details.
	 */
	@Override
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> boolean deleteIsRelatedTo(I isRelatedTo)
			throws IsRelatedToNotFoundException, ResourceRegistryException {
		return delete(isRelatedTo);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #delete(String, UUID)} for implementation details.
	 */
	@Override
	public boolean deleteIsRelatedTo(String isRelatedToType, UUID uuid)
			throws IsRelatedToNotFoundException, ResourceRegistryException {
		return delete(isRelatedToType, uuid);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addToContext(String type, UUID instanceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", type, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());

			
			Map<String, String> queryParams = new HashMap<>();
			queryParams.put(SharingPath.OPERATION_QUERY_PARAMETER, SharingOperation.ADD.name());
			queryParams.put(SharingPath.DRY_RUN_QUERY_QUERY_PARAMETER, dryRun.toString());
			Boolean forceAddToContext = getCurrentContextUUID().compareTo(contextUUID)==0;
			queryParams.put(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, forceAddToContext.toString());
			
			addIncludeMeta(queryParams);
			addIncludeAllMeta(queryParams);
			addIncludeContexts(queryParams);
			gxHTTPStringRequest.queryParams(queryParams);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post();
			String jsonArray = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.info("{} with UUID {} successfully added to {} with UUID {}", type, instanceUUID, Context.NAME, contextUUID);
			List<ERElement> affectedInstaces = ElementMapper.unmarshalList(jsonArray);
			
			return affectedInstaces;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addToContext(ERElement er, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			UUID instanceUUID = er.getID();
			return addToContext(type, instanceUUID, contextUUID, dryRun);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addToCurrentContext(String type, UUID instanceUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(type, instanceUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addToCurrentContext(ERElement er, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(er, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFromContext(String type, UUID instanceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", type, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());
			
			Map<String, String> queryParams = new HashMap<>();
			queryParams.put(SharingPath.OPERATION_QUERY_PARAMETER, SharingOperation.REMOVE.name());
			queryParams.put(SharingPath.DRY_RUN_QUERY_QUERY_PARAMETER, dryRun.toString());
			
			addIncludeMeta(queryParams);
			addIncludeAllMeta(queryParams);
			addIncludeContexts(queryParams);
			gxHTTPStringRequest.queryParams(queryParams);
			
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post();
			String jsonArray = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.info("{} with UUID {} successfully removed from {} with UUID {}", type, instanceUUID, Context.NAME, contextUUID);
			List<ERElement> affectedInstaces = ElementMapper.unmarshalList(jsonArray);
			
			return affectedInstaces;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFromContext(ERElement er, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			String type = er.getTypeName();
			UUID instanceUUID = er.getID();
			return removeFromContext(type, instanceUUID, contextUUID, dryRun);
		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFromCurrentContext(String type, UUID instanceUUID, Boolean dryRun) 
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(type, instanceUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFromCurrentContext(ERElement er, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException,
			ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(er, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<UUID, String> getElementContexts(String type, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to get contexts of {} with UUID {}", type, instanceUUID);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);

			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String objectNode = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.info("Contexts of {} with UUID {} are {}", type, instanceUUID, objectNode);
			
			Map<UUID, String> contexts = ContextUtility.getContextMap(objectNode);

			return contexts;

		} catch (ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch (Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ERElem extends ERElement> Map<UUID, String> getElementContexts(ERElem er)
			throws NotFoundException, ResourceRegistryException {
		String type = er.getTypeName();
		UUID instanceUUID = er.getID();
		return getElementContexts(type, instanceUUID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addResourceToContext(String resourceType, UUID resourceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(resourceType, resourceUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource> List<ERElement> addResourceToContext(R resource, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(resource, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addResourceToCurrentContext(String resourceType, UUID resourceUUID, Boolean dryRun) 
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resourceType, resourceUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource> List<ERElement> addResourceToCurrentContext(R resource, Boolean dryRun) 
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resource, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeResourceFromContext(String resourceType, UUID resourceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(resourceType, resourceUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource> List<ERElement> removeResourceFromContext(R resource, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(resource, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeResourceFromCurrentContext(String resourceType, UUID resourceUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resourceType, resourceUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource> List<ERElement> removeResourceFromCurrentContext(R resource, Boolean dryRun) throws SchemaViolationException,
			ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resource, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<UUID, String> getResourceContexts(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ResourceRegistryException {
		return getElementContexts(resourceType, resourceUUID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource> Map<UUID, String> getResourceContexts(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return getElementContexts(resource);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addFacetToContext(String facetType, UUID facetUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(facetType, facetUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <F extends Facet> List<ERElement> addFacetToContext(F facet, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(facet, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> addFacetToCurrentContext(String facetType, UUID facetUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facetType, facetUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <F extends Facet> List<ERElement> addFacetToCurrentContext(F facet, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facet, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFacetFromContext(String facetType, UUID facetUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(facetType, facetUUID, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <F extends Facet> List<ERElement> removeFacetFromContext(F facet, UUID contextUUID, Boolean dryRun) 
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(facet, contextUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ERElement> removeFacetFromCurrentContext(String facetType, UUID facetUUID, Boolean dryRun) 
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facetType, facetUUID, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <F extends Facet> List<ERElement> removeFacetFromCurrentContext(F facet, Boolean dryRun) 
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facet, dryRun);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<UUID, String> getFacetContexts(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ResourceRegistryException {
		return getElementContexts(facetType, facetUUID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <F extends Facet> Map<UUID, String> getFacetContexts(F facet)
			throws FacetNotFoundException, ResourceRegistryException {
		return getElementContexts(facet);
	}

}
