package org.gcube.informationsystem.resourceregistry.schema;

import java.util.List;

import org.gcube.informationsystem.model.knowledge.ModelKnowledge;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.types.knowledge.TypeInformation;
import org.gcube.informationsystem.types.reference.Type;

/**
 * Client interface for accessing Information System type schemas and model knowledge from the Resource Registry.
 * 
 * <strong>Purpose:</strong>
 * This interface provides comprehensive access to the Information System model definitions, type schemas,
 * hierarchical relationships, and model knowledge within the Resource Registry service. It enables
 * applications to understand the structure and constraints of available entity and relation types.
 * 
 * <strong>Instantiation:</strong>
 * Clients should be created using the factory pattern for proper configuration:
 * <pre>
 * // Recommended way to create a client
 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
 * </pre>
 * 
 * <strong>Supported Operations:</strong>
 * <ul>
 * <li><strong>Schema Access</strong>: Retrieve type definitions, properties, and constraints for all Information System types</li>
 * <li><strong>Model Knowledge</strong>: Access comprehensive model information including type hierarchies and relationships</li>
 * <li><strong>Type Discovery</strong>: Find available types, their inheritance chains, and compatible subtypes</li>
 * <li><strong>Type Validation</strong>: Validate type names and class compatibility</li>
 * <li><strong>Schema Navigation</strong>: Traverse type hierarchies and understand type relationships.</li>
 * </ul>
 * 
 * <strong>Model Knowledge:</strong>
 * The schema client provides access to complete model knowledge including:
 * <ul>
 * <li>Type definitions for Resources, Facets, ConsistsOf relations, and IsRelatedTo relations</li>
 * <li>Property definitions with types, constraints, and validation rules</li>
 * <li>Inheritance hierarchies and polymorphic type relationships</li>
 * <li>Schema versioning and evolution information.</li>
 * </ul>
 * 
 * <strong>Caching Behavior:</strong>
 * The client implements intelligent caching of model knowledge to improve performance:
 * <ul>
 * <li>Model knowledge is cached locally to reduce network requests</li>
 * <li>Cache can be manually refreshed when schema changes are detected</li>
 * <li>Automatic cache invalidation based on schema version changes.</li>
 * </ul>
 * 
 * <strong>Authorization:</strong>
 * Schema access is generally available to all authenticated users, but some advanced schema
 * information may be filtered based on user roles and permissions.
 *
 * <strong>Query Parameter Configuration:</strong>
 * Most methods support optional query parameters that can be configured via the client configuration:
 * <ul>
 * <li><strong>includeMeta</strong>: Include metadata in schema responses.</li>
 * </ul>
 *
 * <strong>Context Support:</strong>
 * The client automatically operates within the context determined by the authorization token.
 *
 * <strong>Thread Safety:</strong>
 * This client is designed to be thread-safe for concurrent access to cached model knowledge
 * and schema information across multiple threads.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistrySchemaClient {

	/**
	 * Returns whether metadata should be included in schema requests.
	 * 
	 * @return true if metadata should be included, false otherwise
	 */
	public boolean includeMeta();

	/**
	 * Sets whether metadata should be included in schema requests.
	 * 
	 * @param includeMeta true to include metadata, false to exclude it
	 */
	public void setIncludeMeta(boolean includeMeta);

	/**
	 * Retrieves the complete model knowledge containing type definitions and their information.
	 * The model knowledge provides access to all registered types and their hierarchical relationships.
	 * 
	 * @return the model knowledge containing type definitions and information
	 */
	public ModelKnowledge<Type, TypeInformation> getModelKnowledge();
	
	/**
	 * Forces a renewal of the cached model knowledge by fetching the latest version from the Resource Registry.
	 * This method should be called when the schema has been modified and the local cache needs to be updated.
	 */
	public void renewModelKnowledge();

	/**
	 * Retrieves the type tree node for a specific type identified by its name.
	 * The tree node contains the type definition and its hierarchical position within the type system.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @return the tree node containing the type and its hierarchical information
	 * @throws SchemaNotFoundException if the specified type name does not exist in the schema
	 * @throws ResourceRegistryException if an error occurs during the operation
	 */
	public Node<Type> getTypeTreeNode(String typeName)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Retrieves the type tree node for a specific type identified by its model element class.
	 * The tree node contains the type definition and its hierarchical position within the type system.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to retrieve
	 * @return the tree node containing the type and its hierarchical information
	 * @throws SchemaNotFoundException if the specified type does not exist in the schema
	 * @throws ResourceRegistryException if an error occurs during the operation
	 */
	public <ME extends ModelElement> Node<Type> getTypeTreeNode(Class<ME> clz)
			throws SchemaNotFoundException, ResourceRegistryException;

	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value Header value
	 */
	public void addHeader(String name, String value);
	
	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value boolean value
	 */
	public void addHeader(String name, boolean value);
	
	/**
	 * Creates a new type definition in the Information System using JSON schema definition.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /types/{type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new type definition from JSON schema representation</li>
	 * <li>Type name is extracted from the JSON schema definition</li>
	 * <li>Validates the JSON schema against Information System type schema requirements</li>
	 * <li>Type definitions become available system-wide for instance creation</li>
	 * <li>Returns the complete type definition including metadata and schema information.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Type successfully created</li>
	 * <li><strong>409 Conflict</strong>: Type already exists with the given name</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON schema or malformed request</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to create types.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>JSON schema must conform to Information System type schema format</li>
	 * <li>Type name must be unique in the system</li>
	 * <li>Schema must define valid properties, relationships, and constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All type properties and relationships are validated against the Information System model</li>
	 * <li>Type hierarchy and inheritance rules are enforced</li>
	 * <li>Property types and constraints are validated</li>
	 * <li>Relationship definitions must reference valid types.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can create any type definition without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can create any type definition without restrictions</li>
	 * <li><strong>All Other Users</strong>: Cannot create type definitions (will receive authorization errors)</li>
	 * <li>Type creation is restricted to prevent system-wide compatibility issues.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 *
	 * // Create a new type from JSON schema
	 * String jsonSchema = "{ \"@class\": \"ContactFacet\", ... }";
	 * String createdType = client.create(jsonSchema);
	 * </pre>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Type definitions are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * @param typeDefinitition the JSON schema definition of the type to create
	 * @return the JSON representation of the created type definition
	 * @throws SchemaException if the provided schema is invalid or malformed
	 * @throws ResourceRegistryException for creation errors or authorization failures
	 */
	public String create(String typeDefinitition) throws SchemaException, ResourceRegistryException;
	
	/**
	 * Creates a new type definition in the Information System using a model element class.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /types/{type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new type definition derived from the provided model element class</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>Analyzes class annotations and structure to generate the type schema</li>
	 * <li>Type definitions become available system-wide for instance creation</li>
	 * <li>Returns the complete Type object with metadata and schema information.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Type successfully created</li>
	 * <li><strong>409 Conflict</strong>: Type already exists with the given name</li>
	 * <li><strong>400 Bad Request</strong>: Invalid class definition or schema generation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to create types.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Class must extend ModelElement or its subtypes</li>
	 * <li>Annotations are processed to generate schema constraints</li>
	 * <li>Property definitions are extracted from class fields and methods.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All class properties and relationships are validated against the Information System model</li>
	 * <li>Type hierarchy and inheritance rules are enforced</li>
	 * <li>Property types and constraints are validated from class annotations</li>
	 * <li>Relationship definitions must reference valid types.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can create any type definition without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can create any type definition without restrictions</li>
	 * <li><strong>All Other Users</strong>: Cannot create type definitions (will receive authorization errors)</li>
	 * <li>Type creation is restricted to prevent system-wide compatibility issues.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Create a new type from model element class
	 * Type createdType = client.create(ContactFacet.class);
	 * </pre>
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to create
	 * @return the Type object representing the created type definition
	 * @throws SchemaException if the class definition is invalid or schema generation fails
	 * @throws ResourceRegistryException for creation errors or authorization failures
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Type definitions are global and not bound to a specific context.</li>
	 * </ul>
	 */
	public <ME extends ModelElement> Type create(Class<ME> clz) throws SchemaException, ResourceRegistryException;

	/**
	 * Checks if a type definition exists in the Information System using the type name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified type definition using its name</li>
	 * <li>Does not return the type data, only confirms existence</li>
	 * <li>Checks accessibility within the current user authorization level</li>
	 * <li>Most direct method when type name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Type does not exist in the system</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>All Other Users</strong>: Can check existence of any type definition with filtered access to sensitive metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Check if ContactFacet type exists
	 * boolean contactFacetExists = client.exist("ContactFacet");
	 * 
	 * // Check if a custom resource type exists
	 * boolean customResourceExists = client.exist("MyCustomResource");
	 * </pre>
	 * 
	 * @param typeName the name of the type to check for existence (case-sensitive)
	 * @return true if the type exists in the system, false if not found
	 * @throws ResourceRegistryException if an error occurs during the existence check
	 */
	public boolean exist(String typeName) throws ResourceRegistryException;

	/**
	 * Checks if a type definition exists in the Information System using a model element class.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified type definition using its model element class</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>Does not return the type data, only confirms existence</li>
	 * <li>Checks accessibility within the current user authorization level</li>
	 * <li>Convenient method when working with strongly-typed classes.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Type corresponding to the class does not exist in the system</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Class must extend ModelElement or its subtypes</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>All Other Users</strong>: Can check existence of any type definition with filtered access to sensitive metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Check if ContactFacet type exists
	 * boolean contactFacetExists = client.exist(ContactFacet.class);
	 * 
	 * // Check if EService type exists
	 * boolean eServiceExists = client.exist(EService.class);
	 * </pre>
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to check for existence
	 * @return true if the type corresponding to the class exists in the system, false if not found
	 * @throws ResourceRegistryException if an error occurs during the existence check
	 */
	public <ME extends ModelElement> boolean exist(Class<ME> clz) throws ResourceRegistryException;

	/**
	 * Reads the schema definition for a specified type by name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves type schema definition(s) for the specified type name</li>
	 * <li>When polymorphic=false: returns only the specified type definition</li>
	 * <li>When polymorphic=true: returns the specified type AND all its existing subtypes</li>
	 * <li>Metadata inclusion is controlled by client configuration and user authorization</li>
	 * <li>Response detail level varies based on user authorization (IS/Infrastructure managers get full metadata).</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Type with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (parameter of this method):
	 * <ul>
	 * <li>Whether to include subtypes in the response</li>
	 * <li>Default value: false (returns only the specified type)</li>
	 * <li>Values: true (includes all subtypes) | false (specified type only)</li>
	 * <li>When true: returns the specified type AND all its existing subtypes</li>
	 * <li>When false: returns only the specified type definition</li>
	 * <li>Note: No pagination is applied - ALL subtypes are returned when polymorphic=true.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response type definition</li>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.TypePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>Infrastructure-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>All Other Users</strong>: Receive basic type information with filtered metadata (sensitive information obfuscated).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Read a ContactFacet type definition only
	 * String contactFacetSchema = client.read("ContactFacet", false);
	 * 
	 * // Read ContactFacet and all its subtypes
	 * String contactFacetAndSubtypes = client.read("ContactFacet", true);
	 * </pre>
	 * 
	 * @param typeName the name of the type to retrieve (case-sensitive)
	 * @param polymorphic whether to include subtypes in the response (when true, includes all subtypes)
	 * @return JSON string containing type schema definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public String read(String typeName, Boolean polymorphic) throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Reads the type definitions for a specified model element class.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves type definitions for the specified model element class</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>When polymorphic=false: returns only the specified type definition</li>
	 * <li>When polymorphic=true: returns the specified type AND all its existing subtypes</li>
	 * <li>Returns strongly-typed Type objects instead of JSON strings</li>
	 * <li>Response detail level varies based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Type corresponding to the specified class does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (parameter of this method):
	 * <ul>
	 * <li>Whether to include subtypes in the response</li>
	 * <li>Default value: false (returns only the specified type)</li>
	 * <li>Values: true (includes all subtypes) | false (specified type only)</li>
	 * <li>When true: returns the specified type AND all its existing subtypes</li>
	 * <li>When false: returns only the specified type definition</li>
	 * <li>Note: No pagination is applied - ALL subtypes are returned when polymorphic=true.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response type definition</li>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.TypePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>Infrastructure-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>All Other Users</strong>: Receive basic type information with filtered metadata (sensitive information obfuscated).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Read ContactFacet type definition only
	 * List&lt;Type&gt; contactFacetType = client.read(ContactFacet.class, false);
	 * 
	 * // Read ContactFacet and all its subtypes
	 * List&lt;Type&gt; contactFacetAndSubtypes = client.read(ContactFacet.class, true);
	 * </pre>
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to retrieve
	 * @param polymorphic whether to include subtypes in the response (when true, includes all subtypes)
	 * @return list of Type objects representing the type definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the type corresponding to the specified class is not found
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public <ME extends ModelElement> List<Type> read(Class<ME> clz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException;

	/**
	 * Reads the schema definition for a specified type by name up to a specific hierarchy level.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves type schema definition for the specified type name</li>
	 * <li>Limits the depth of hierarchical relationships to the specified level</li>
	 * <li>Level parameter controls how deep into related types the response includes</li>
	 * <li>Useful for preventing overly complex responses in deeply nested type hierarchies</li>
	 * <li>Response detail level varies based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Type with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>level</strong> (parameter of this method):
	 * <ul>
	 * <li>Controls the depth of hierarchical relationships included in the response</li>
	 * <li>Higher values include more detailed relationship information</li>
	 * <li>Lower values provide more concise responses with limited relationship depth</li>
	 * <li>Useful for controlling response size and complexity.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response type definition</li>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.TypePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>Infrastructure-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>All Other Users</strong>: Receive basic type information with filtered metadata (sensitive information obfuscated).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Read ContactFacet type definition with hierarchy level 1
	 * String contactFacetSchema = client.read("ContactFacet", 1);
	 * 
	 * // Read ContactFacet type definition with deeper hierarchy level
	 * String contactFacetDeepSchema = client.read("ContactFacet", 3);
	 * </pre>
	 * 
	 * @param typeName the name of the type to retrieve (case-sensitive)
	 * @param level the maximum hierarchy level to include in the response
	 * @return JSON string containing type schema definition with hierarchy limited to the specified level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public String read(String typeName, int level)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Reads the type definitions for a specified model element class up to a specific hierarchy level.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves type definitions for the specified model element class</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>Limits the depth of hierarchical relationships to the specified level</li>
	 * <li>Level parameter controls how deep into related types the response includes</li>
	 * <li>Returns strongly-typed Type objects instead of JSON strings</li>
	 * <li>Response detail level varies based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Type corresponding to the specified class does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>level</strong> (parameter of this method):
	 * <ul>
	 * <li>Controls the depth of hierarchical relationships included in the response</li>
	 * <li>Higher values include more detailed relationship information</li>
	 * <li>Lower values provide more concise responses with limited relationship depth</li>
	 * <li>Useful for controlling response size and complexity.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response type definition</li>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.TypePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>Infrastructure-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>All Other Users</strong>: Receive basic type information with filtered metadata (sensitive information obfuscated).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistrySchemaClient client = ResourceRegistrySchemaClientFactory.create();
	 * 
	 * // Read ContactFacet type definition with hierarchy level 1
	 * List&lt;Type&gt; contactFacetType = client.read(ContactFacet.class, 1);
	 * 
	 * // Read ContactFacet type definition with deeper hierarchy level
	 * List&lt;Type&gt; contactFacetDeepType = client.read(ContactFacet.class, 3);
	 * </pre>
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to retrieve
	 * @param level the maximum hierarchy level to include in the response
	 * @return list of Type objects representing the type definition(s) with hierarchy limited to the specified level
	 * @throws SchemaNotFoundException if the type corresponding to the specified class is not found
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public <ME extends ModelElement> List<Type> read(Class<ME> clz, int level)
			throws SchemaNotFoundException, ResourceRegistryException;
	
}
