package org.gcube.informationsystem.resourceregistry.rest;

import java.util.UUID;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.enums.ParameterIn;
import org.eclipse.microprofile.openapi.annotations.enums.SchemaType;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.media.Schema;
import org.eclipse.microprofile.openapi.annotations.parameters.Parameter;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.BaseRequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.entities.ContextManagement;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;
import org.gcube.smartgears.utils.InnerMethodName;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.PATCH;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(ContextPath.CONTEXTS_PATH_PART)
@Tag(name = "Contexts", description = "Operations for managing contexts in the Resource Registry.")
public class ContextManager extends BaseRest {
	
		/**
	 * Constant representing the context UUID path parameter.
	 */
	public static final String CONTEXT_UUID_PATH_PARAMETER = "context-uuid";

	public ContextManager() {
		super();
	}
	
	/**
	 * Retrieves the list of all contexts in the system.
	 * The response can include metadata and provides pagination based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /contexts[?limit={limit}&offset={offset}&includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>GET /contexts (returns first 10 contexts - default pagination)</li>
	 * <li>GET /contexts?limit=20 (returns first 20 contexts)</li>
	 * <li>GET /contexts?offset=10 (returns 10 contexts starting from the 11th - default limit with offset)</li>
	 * <li>GET /contexts?limit=5&amp;offset=15 (returns 5 contexts starting from the 16th)</li>
	 * <li>GET /contexts?includeMeta=true (returns first 10 contexts with metadata - for non-admin users)</li>
	 * <li>GET /contexts?limit=20&amp;includeMeta=true (returns first 20 contexts with metadata).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of contexts to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 contexts).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of contexts to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first context)</li>
	 * <li>Example: ?offset=100 (skips the first 100 contexts)</li>
	 * <li>Commonly used with limit for pagination: ?limit=50&amp;offset=100.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields)</li>
	 * <li>Note: IS-Manager and Infrastructure-Manager always receive metadata regardless of this parameter.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Automatically receive full context information including:</li>
	 * </ul>
	 *   • Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
	 *   • Context state information;
	 *   • All context properties without obfuscation.
	 * <ul>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Automatically receive full context information including:</li>
	 * </ul>
	 *   • Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
	 *   • Context state information;
	 *   • All context properties without obfuscation.
	 * <ul>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Receive basic context information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 * </ul>
	 *   • Date fields (creation, modification) are visible;
	 *   • User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
	 *   • Other administrative details may be filtered.
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Contexts successfully retrieved</li>
	 * <li>403 Forbidden: User lacks authorization to access context information</li>
	 * <li>500 Internal Server Error: Server error during context retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: JSON array containing context objects with role-appropriate detail level</li>
	 * <li>Pagination information may be included in response headers.</li>
	 * </ul>
	 * 
	 * @return JSON array containing context objects with role-appropriate detail level
	 * @throws ContextNotFoundException if contexts cannot be retrieved
	 * @throws ResourceRegistryException for other retrieval errors
	 */
	@GET
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Contexts",
		description = """
		Retrieves the list of contexts in the system.

		The response can include metadata and provides pagination based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/contexts (returns first 10 contexts - default pagination);
		- GET /access/contexts?limit=20 (returns first 20 contexts);
		- GET /access/contexts?offset=10 (returns 10 contexts starting from the 11th - default limit with offset);
		- GET /access/contexts?limit=5&offset=15 (returns 5 contexts starting from the 16th);
		- GET /access/contexts?includeMeta=true (returns first 10 contexts with metadata - for non-admin users);
		- GET /access/contexts?limit=20&includeMeta=true (returns first 20 contexts with metadata).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Automatically receive full context information including:
				- Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
				- Context state information;
				- All context properties without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Infrastructure-Manager:**
			- Automatically receive full context information including:
				- Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
				- Context state information;
				- All context properties without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Other Users:**
			- Receive basic context information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)
	@Parameter(
		name = ContextPath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of contexts to return in a single response.
		- Default: 10 (returns at most 10 contexts);
		- Example: ?limit=50 (returns at most 50 contexts).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = ContextPath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of contexts to skip from the beginning of the result set.
		- Default: 0 (starts from the first context);
		- Example: ?offset=100 (skips the first 100 contexts);
		- Commonly used with limit for pagination: ?limit=50&offset=100.
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = ContextPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response (for non-admin users). 
		- Default: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields);
		- Note: IS-Manager and Infrastructure-Manager always receive not filtered metadata regardless of this parameter.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Contexts successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access context information"
	)
	@APIResponse(
		responseCode = "500",
		description = "Server error during context retrieval"
	)
	public String listContexts() throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to read all {}s", Context.NAME);
		setAccountingMethod(Method.LIST, Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo(BaseRequestInfo.DEFAULT_OFFSET, BaseRequestInfo.UNBOUNDED_LIMIT);
		// TODO check if the user has the role to request such parameters
		serverRequestInfo.setIncludeMeta(true);
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkLimitOffset();
		
		ContextManagement contextManagement = new ContextManagement();
		return contextManagement.all(false);
	}
	
	/**
	 * Retrieves a specific context by its UUID.
	 * The response includes metadata and detail level based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>GET /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (returns basic context information)</li>
	 * <li>GET /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0?includeMeta=true (returns context with metadata - for non-admin users)</li>
	 * <li>GET /contexts/CURRENT_CONTEXT (returns the current context - retrieved from client's authorization token).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * 
	 * <strong>context-uuid</strong> (required):
	 * <ul>
	 * <li>The unique identifier of the context to retrieve</li>
	 * <li>Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0")</li>
	 * <li><strong>Special keyword</strong>: "CURRENT_CONTEXT" to retrieve the context from client's authorization token</li>
	 * <li>When using "CURRENT_CONTEXT", the actual context is extracted from the JWT token sent by the client</li>
	 * <li>Example: GET /contexts/CURRENT_CONTEXT returns the context the client is authorized for.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields)</li>
	 * <li>Note: IS-Manager and Infrastructure-Manager always receive metadata regardless of this parameter.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any context without restrictions</li>
	 * <li>Automatically receive full context information including complete metadata</li>
	 * <li>All context properties and state information without obfuscation</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any context without restrictions</li>
	 * <li>Automatically receive full context information including complete metadata</li>
	 * <li>All context properties and state information without obfuscation</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>When operating in their own context (using a token from that context):</li>
	 * </ul>
	 *   • Receive full context information like IS-Manager and Infrastructure-Manager;
	 *   • Complete metadata without obfuscation;
	 *   • All context properties and state information.
	 * <ul>
	 * <li>When operating outside their context or requesting other contexts:</li>
	 * </ul>
	 *   • Receive basic context information by default;
	 *   • Can request metadata via includeMeta=true with obfuscated sensitive fields.
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Receive basic context information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 * </ul>
	 *   • Date fields (creation, modification) are visible;
	 *   • User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
	 *   • Other administrative details may be filtered.
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Context successfully retrieved</li>
	 * <li>404 Not Found: Context with the specified UUID does not exist</li>
	 * <li>403 Forbidden: User lacks authorization to access the specific context</li>
	 * <li>400 Bad Request: Invalid UUID format.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: JSON object containing the context with role-appropriate detail level</li>
	 * <li>Includes context properties, metadata (if authorized), and state information.</li>
	 * </ul>
	 * 
	 * @param uuid the UUID of the context to retrieve, or "CURRENT_CONTEXT" to get the context from client's authorization token
	 * @return JSON object containing the context with role-appropriate detail level
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException for other retrieval errors
	 */
	@GET
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Get Context by UUID",
		description = """
		Retrieves a specific context by its UUID.

		The response includes metadata and detail level based on query parameters and user authorization.


		**Request Examples:**
		- GET /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (returns basic context information);
		- GET /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0?includeMeta=true (returns context with metadata - for non-admin users);
		- GET /contexts/CURRENT_CONTEXT (returns the current context - retrieved from client's authorization token).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can retrieve context information for any context without restrictions;
			- Automatically receive full context information including complete metadata;
			- All context properties and state information without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Infrastructure-Manager:**
			- Can retrieve context information for any context without restrictions;
			- Automatically receive full context information including complete metadata;
			- All context properties and state information without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Context-Manager:**
			- When operating in their own context (using a token from that context):
				- Receive full context information like IS-Manager and Infrastructure-Manager;
				- Complete metadata without obfuscation;
				- All context properties and state information.
			- When operating outside their context or requesting other contexts:
				- Receive basic context information by default;
				- Can request metadata via includeMeta=true with obfuscated sensitive fields.

		- **Other Users:**
			- Receive basic context information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)
	@Parameter(
		name = ContextPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response (for non-admin users).
		- Default: false (basic information only);
		- Values: true|false;
		- Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields);
		- Note: IS-Manager and Infrastructure-Manager always receive metadata regardless of this parameter.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Context successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Context with the specified UUID does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access the specific context"
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid UUID format"
	)
	public String read(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER)
			@Parameter(
				name = ContextManager.CONTEXT_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The unique identifier of the context to retrieve.
				- Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0");
				- Special keyword: "CURRENT_CONTEXT" to retrieve the context from client's authorization token;
				- When using "CURRENT_CONTEXT", the actual context is extracted from the JWT token sent by the client;
				- Example: GET /contexts/CURRENT_CONTEXT returns the context the client is authorized for.
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "c0f314e7-2807-4241-a792-2a6c79ed4fd0")
			)
			String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		if(uuid.compareTo(ContextPath.CURRENT_CONTEXT_PATH_PART)==0){
			uuid = ContextUtility.getCurrentRequestEnvironment().getUUID().toString();
		}
		logger.info("Requested to read {} with id {} ", Context.NAME, uuid);
		setAccountingMethod(Method.READ, Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setIncludeMeta(true);
		serverRequestInfo.setAllMeta(true);
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		return contextManagement.readAsString();
	}
	
	/**
	 * Creates a new context or updates an existing context with the provided JSON representation.
	 * Only authorized administrative users can perform this operation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /contexts/{context-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>PUT /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (create/update context).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * 
	 * <strong>context-uuid</strong> (required):
	 * <ul>
	 * <li>The unique identifier of the context to create or update</li>
	 * <li>Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON representation of the context to create or update</li>
	 * <li>Must conform to the Context schema definition.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can create/update any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can create/update any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can create/update contexts only if they are Context-Manager of the parent context</li>
	 * <li>The parent context must be the current context (retrieved from client's authorization token)</li>
	 * <li>Cannot create/update contexts outside their management scope.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Cannot create or update contexts</li>
	 * <li>Will receive authorization errors if attempting these operations.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * 
	 * <strong>Context Property Changes and Operations:</strong>
	 * 
	 * <strong>Name Change:</strong>
	 * <ul>
	 * <li>Updates the context name while preserving all other properties</li>
	 * <li>Changes the full path of the context and all its child contexts</li>
	 * <li>No effect on instances belonging to this context (instances remain in the context)</li>
	 * <li>No effect on instances in child contexts.</li>
	 * </ul>
	 * 
	 * <strong>Parent Change (Move Operation):</strong>
	 * <ul>
	 * <li>Moves the context to a different parent in the context hierarchy</li>
	 * <li>Changes the full path of the context and all its child contexts</li>
	 * <li>No effect on instances belonging to this context (instances remain in the context)</li>
	 * <li>No effect on instances in child contexts</li>
	 * <li>However, instances in hierarchical queries of the old and new parent contexts will change.</li>
	 * </ul>
	 * 
	 * <strong>Combined Name and Parent Change:</strong>
	 * <ul>
	 * <li>Both name and parent can be changed in the same update request</li>
	 * <li>Results in both a rename and move operation applied simultaneously</li>
	 * <li>The full path changes due to both the new name and new parent location</li>
	 * <li>All effects described above for both operations apply.</li>
	 * </ul>
	 * 
	 * <strong>Additional Properties:</strong>
	 * <ul>
	 * <li>While certain roles may receive additional parameters in context definitions, these cannot be modified through update operations</li>
	 * <li>Only name and parent can be effectively changed via update.</li>
	 * </ul>
	 * 
	 * <strong>Important Design Considerations:</strong>
	 * <ul>
	 * <li>The service is designed to work with UUIDs rather than full paths precisely because full paths can change</li>
	 * <li>Context UUIDs remain stable regardless of name or parent changes</li>
	 * <li>All internal references and relationships use UUIDs for consistency</li>
	 * <li>Client applications should store and reference contexts by UUID, not by full path.</li>
	 * </ul>
	 * 
	 * <strong>Hierarchical Query Impact:</strong>
	 * <ul>
	 * <li>When a context is moved to a new parent, the hierarchical queries (queries that include child contexts) in both the old and new parent contexts will reflect the change</li>
	 * <li><strong>Instances</strong> belonging to the moved context will no longer appear in hierarchical queries of the old parent context</li>
	 * <li><strong>Instances</strong> belonging to the moved context will start appearing in hierarchical queries of the new parent context</li>
	 * <li>The context hierarchy structure itself changes, but the effect is visible in terms of which instances are returned when querying with hierarchical=true parameter</li>
	 * <li><strong>Context Tree Structure</strong>: When listing contexts to construct the context tree hierarchy, the tree structure will obviously change to reflect the new parent-child relationships.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Context successfully updated</li>
	 * <li>201 Created: Context successfully created</li>
	 * <li>400 Bad Request: Invalid JSON representation or malformed request</li>
	 * <li>403 Forbidden: User lacks authorization to create/update contexts</li>
	 * <li>404 Not Found: Parent context not found (when creating new context).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: Complete context information including metadata</li>
	 * <li>All context properties and state information without obfuscation.</li>
	 * </ul>
	 * 
	 * @param uuid the UUID of the context to create or update
	 * @param json the JSON representation of the context to create or update
	 * @return JSON object containing the created/updated context with complete metadata
	 * @throws ResourceRegistryException for creation/update errors or authorization failures
	 */
	@PUT
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Create or Update Context",
		description = """
		Creates a new context or updates an existing context with the provided JSON representation.

		Only authorized administrative users can perform this operation.


		**Request Examples:**
		- PUT /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (create/update context).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can create/update any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Infrastructure-Manager:**
			- Can create/update any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Context-Manager:**
			- Can create/update contexts only if they are Context-Manager of the parent context;
			- The parent context must be the current context (retrieved from client's authorization token);
			- Cannot create/update contexts outside their management scope.

		- **Other Users:**
			- Cannot create or update contexts;
			- Will receive authorization errors if attempting these operations.


		**Operation Behavior:**

		**Context Property Changes and Operations:**

		**Name Change:**
		- Updates the context name while preserving all other properties;
		- Changes the full path of the context and all its child contexts;
		- No effect on instances belonging to this context (instances remain in the context);
		- No effect on instances in child contexts.

		**Parent Change (Move Operation):**
		- Moves the context to a different parent in the context hierarchy;
		- Changes the full path of the context and all its child contexts;
		- No effect on instances belonging to this context (instances remain in the context);
		- No effect on instances in child contexts;
		- However, instances in hierarchical queries of the old and new parent contexts will change.

		**Combined Name and Parent Change:**
		- Both name and parent can be changed in the same update request;
		- Results in both a rename and move operation applied simultaneously;
		- The full path changes due to both the new name and new parent location;
		- All effects described above for both operations apply.

		**Additional Properties:**
		- While certain roles may receive additional parameters in context definitions, these cannot be modified through update operations;
		- Only name and parent can be effectively changed via update.

		**Important Design Considerations:**
		- The service is designed to work with UUIDs rather than full paths precisely because full paths can change;
		- Context UUIDs remain stable regardless of name or parent changes;
		- All internal references and relationships use UUIDs for consistency;
		- Client applications should store and reference contexts by UUID, not by full path.

		**Hierarchical Query Impact:**
		- When a context is moved to a new parent, the hierarchical queries (queries that include child contexts) in both the old and new parent contexts will reflect the change;
		- **Instances** belonging to the moved context will no longer appear in hierarchical queries of the old parent context;
		- **Instances** belonging to the moved context will start appearing in hierarchical queries of the new parent context;
		- The context hierarchy structure itself changes, but the effect is visible in terms of which instances are returned when querying with hierarchical=true parameter;
		- **Context Tree Structure**: When listing contexts to construct the context tree hierarchy, the tree structure will obviously change to reflect the new parent-child relationships.
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Context successfully updated",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "201",
		description = "Context successfully created",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid JSON representation or malformed request"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to create/update contexts"
	)
	@APIResponse(
		responseCode = "404",
		description = "Parent context not found (when creating new context)"
	)
	public String updateCreate(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER)
			@Parameter(
				name = ContextManager.CONTEXT_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The unique identifier of the context to create or update.
				- Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "c0f314e7-2807-4241-a792-2a6c79ed4fd0")
			)
			String uuid, String json)
			throws ResourceRegistryException {
		logger.info("Requested to update/create {} with json {} ", Context.NAME, json);
		setAccountingMethod(Method.UPDATE, Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setIncludeMeta(true);
		serverRequestInfo.setAllMeta(true);
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		contextManagement.setJson(json);
		return contextManagement.createOrUpdate();
	}
	
	/**
	 * Changes the state of an existing context.
	 * Only the state property from the JSON body is evaluated and returns the complete updated context.
	 * Only authorized administrative users can perform this operation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PATCH /contexts/{context-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>PATCH /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 </li>
	 * </ul>
	 *   Body: {"state": "active"} (changes context state to active).
	 * 
	 * <strong>Path Parameters:</strong>
	 * 
	 * <strong>context-uuid</strong> (required):
	 * <ul>
	 * <li>The unique identifier of the context whose state should be changed</li>
	 * <li>Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON representation containing the new state (other properties ignored)</li>
	 * <li>Must contain a Context JSON representation</li>
	 * <li>Only the "state" property will be processed and applied.</li>
	 * </ul>
	 * 
	 * <strong>Body Requirements:</strong>
	 * 
	 * <strong>JSON Body Structure:</strong>
	 * <ul>
	 * <li>Must contain a Context JSON representation</li>
	 * <li>Only the "state" property will be processed and applied</li>
	 * <li>All other properties in the body are ignored</li>
	 * <li>State values must be valid according to ContextState class.</li>
	 * </ul>
	 * 
	 * <strong>Valid State Values:</strong>
	 * <ul>
	 * <li>Refer to ContextState class for complete list of valid state values</li>
	 * <li>Invalid state values will result in an error.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can change state of any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can change state of any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can change state of contexts only if they are Context-Manager of the parent context</li>
	 * <li>The parent context must be the current context (retrieved from client's authorization token)</li>
	 * <li>Cannot change state of contexts outside their management scope.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Cannot change context states</li>
	 * <li>Will receive authorization errors if attempting these operations.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Context state successfully changed</li>
	 * <li>400 Bad Request: Invalid state value or malformed request</li>
	 * <li>403 Forbidden: User lacks authorization to change context state</li>
	 * <li>404 Not Found: Context with specified UUID does not exist.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: Complete updated context (not just the state)</li>
	 * <li>Full context information including complete metadata without obfuscation</li>
	 * <li>All context properties and updated state information are included</li>
	 * <li>Metadata is always included since only administrative users can access this endpoint.</li>
	 * </ul>
	 * 
	 * @param uuid the UUID of the context whose state should be changed
	 * @param json the JSON representation containing the new state (other properties ignored)
	 * @return JSON object containing the complete updated context with new state and full metadata
	 * @throws ResourceRegistryException for state change errors or authorization failures
	 */
	@PATCH
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Change Context State",
		description = """
		Changes the state of an existing context.

		Only the state property from the JSON body is evaluated and returns the complete updated context.
		Only authorized administrative users can perform this operation.


		**Request Examples:**
		- PATCH /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 
		  Body: {"state": "active"} (changes context state to active).


		**Body Requirements:**

		**JSON Body Structure:**
		- Must contain a Context JSON representation;
		- Only the "state" property will be processed and applied;
		- All other properties in the body are ignored;
		- State values must be valid according to ContextState class.

		**Valid State Values:**
		- Refer to ContextState class for complete list of valid state values;
		- Invalid state values will result in an error.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can change state of any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Infrastructure-Manager:**
			- Can change state of any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Context-Manager:**
			- Can change state of contexts only if they are Context-Manager of the parent context;
			- The parent context must be the current context (retrieved from client's authorization token);
			- Cannot change state of contexts outside their management scope.

		- **Other Users:**
			- Cannot change context states;
			- Will receive authorization errors if attempting these operations.
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Context state successfully changed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid state value or malformed request"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to change context state"
	)
	@APIResponse(
		responseCode = "404",
		description = "Context with specified UUID does not exist"
	)
	public String changeState(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER)
			@Parameter(
				name = ContextManager.CONTEXT_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The unique identifier of the context whose state should be changed.
				- Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "c0f314e7-2807-4241-a792-2a6c79ed4fd0")
			)
			String uuid, String json)
			throws ResourceRegistryException {
		logger.info("Requested to activate {} with UUID {}", Context.NAME, uuid);
		setAccountingMethod(Method.UPDATE, Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setIncludeMeta(true);
		serverRequestInfo.setAllMeta(true);
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setJson(json);
		contextManagement.setUUID(UUID.fromString(uuid));
		return contextManagement.changeState();
	}
	
	/**
	 * Deletes an existing context permanently from the system.
	 * Only authorized administrative users can perform this irreversible operation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code DELETE /contexts/{context-uuid}}
	 * 
	 * <strong>IMPORTANT: This is an irreversible operation that permanently removes the context from the system.</strong>
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>DELETE /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (delete the specified context).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * 
	 * <strong>context-uuid</strong> (required):
	 * <ul>
	 * <li>The unique identifier of the context to delete</li>
	 * <li>Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any context without restrictions</li>
	 * <li>Has full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete contexts only if they are Context-Manager of the parent context</li>
	 * <li>The parent context must be the current context (retrieved from client's authorization token)</li>
	 * <li>Cannot delete contexts outside their management scope.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Cannot delete contexts</li>
	 * <li>Will receive authorization errors if attempting these operations.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * 
	 * <strong>Deletion Restrictions and Behavior:</strong>
	 * 
	 * <strong>Context Hierarchy Restrictions:</strong>
	 * <ul>
	 * <li>Cannot delete a context that has child contexts</li>
	 * <li>All child contexts must be deleted first (or moved to a different parent) before deleting a parent context</li>
	 * <li>This ensures the integrity of the context hierarchy structure.</li>
	 * </ul>
	 * 
	 * <strong>Instance Management:</strong>
	 * <ul>
	 * <li>Instances belonging to the deleted context will no longer belong to any context</li>
	 * <li><strong>Critical</strong>: If instances exist only in the deleted context (not shared with other contexts), they will be permanently deleted from the system</li>
	 * <li>Instances that also belong to other contexts will remain in those other contexts</li>
	 * <li><strong>This instance deletion is irreversible</strong> - ensure instances are backed up or moved to other contexts if needed.</li>
	 * </ul>
	 * 
	 * <strong>Hierarchical Query Impact:</strong>
	 * <ul>
	 * <li>Hierarchical queries performed on the parent context will change</li>
	 * <li>Instances that were previously returned in hierarchical queries (due to belonging to the deleted context) will no longer appear</li>
	 * <li>The context tree structure will be updated to reflect the removal.</li>
	 * </ul>
	 * 
	 * <strong>Context History and Audit Trail:</strong>
	 * <ul>
	 * <li>While the context is permanently removed from the active system, a record is maintained in an internal "cemetery" system</li>
	 * <li>This allows for audit trails and historical tracking of deleted contexts</li>
	 * <li><strong>Future Enhancement</strong>: Administrative users will have the ability to list deleted contexts from this cemetery</li>
	 * <li>The UUID of the deleted context becomes permanently unavailable for reuse.</li>
	 * </ul>
	 * 
	 * <strong>Irreversible Operation:</strong>
	 * <ul>
	 * <li><strong>This operation cannot be undone</strong></li>
	 * <li>Once deleted, the context cannot be restored through normal operations</li>
	 * <li>All relationships, instances, and configuration associated with the context are permanently lost</li>
	 * <li>Ensure proper backup and confirmation procedures before performing this operation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>204 No Content: Context successfully deleted</li>
	 * <li>400 Bad Request: Context has child contexts and cannot be deleted</li>
	 * <li>403 Forbidden: User lacks authorization to delete contexts</li>
	 * <li>404 Not Found: Context with specified UUID does not exist.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: N/A (no response body)</li>
	 * <li>Body: Empty (204 No Content response)</li>
	 * <li>No response body is returned upon successful deletion</li>
	 * <li>The context is permanently removed from the system.</li>
	 * </ul>
	 * 
	 * @param uuid the UUID of the context to delete
	 * @return HTTP 204 No Content response upon successful deletion
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException for deletion errors or authorization failures
	 */
	@DELETE
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Operation(
		summary = "Delete Context",
		description = """
		Deletes an existing context permanently from the system.

		Only authorized administrative users can perform this irreversible operation.

		**IMPORTANT: This is an irreversible operation that permanently removes the context from the system.**


		**Request Examples:**
		- DELETE /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (delete the specified context).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can delete any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Infrastructure-Manager:**
			- Can delete any context without restrictions;
			- Has full administrative privileges across all contexts.

		- **Context-Manager:**
			- Can delete contexts only if they are Context-Manager of the parent context;
			- The parent context must be the current context (retrieved from client's authorization token);
			- Cannot delete contexts outside their management scope.

		- **Other Users:**
			- Cannot delete contexts;
			- Will receive authorization errors if attempting these operations.


		**Operation Behavior:**

		**Deletion Restrictions and Behavior:**

		**Context Hierarchy Restrictions:**
		- Cannot delete a context that has child contexts;
		- All child contexts must be deleted first (or moved to a different parent) before deleting a parent context;
		- This ensures the integrity of the context hierarchy structure.

		**Instance Management:**
		- Instances belonging to the deleted context will no longer belong to any context;
		- **Critical**: If instances exist only in the deleted context (not shared with other contexts), they will be permanently deleted from the system;
		- Instances that also belong to other contexts will remain in those other contexts;
		- **This instance deletion is irreversible** - ensure instances are backed up or moved to other contexts if needed.

		**Hierarchical Query Impact:**
		- Hierarchical queries performed on the parent context will change;
		- Instances that were previously returned in hierarchical queries (due to belonging to the deleted context) will no longer appear;
		- The context tree structure will be updated to reflect the removal.

		**Context History and Audit Trail:**
		- While the context is permanently removed from the active system, a record is maintained in an internal "cemetery" system;
		- This allows for audit trails and historical tracking of deleted contexts;
		- **Future Enhancement**: Administrative users will have the ability to list deleted contexts from this cemetery;
		- The UUID of the deleted context becomes permanently unavailable for reuse.

		**Irreversible Operation:**
		- **This operation cannot be undone**;
		- Once deleted, the context cannot be restored through normal operations;
		- All relationships, instances, and configuration associated with the context are permanently lost;
		- Ensure proper backup and confirmation procedures before performing this operation.
		"""
	)
	@APIResponse(
		responseCode = "204",
		description = "Context successfully deleted"
	)
	@APIResponse(
		responseCode = "400",
		description = "Context has child contexts and cannot be deleted"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to delete contexts"
	)
	@APIResponse(
		responseCode = "404",
		description = "Context with specified UUID does not exist"
	)
	public Response delete(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER)
			@Parameter(
				name = ContextManager.CONTEXT_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The unique identifier of the context to delete.
				- Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0").
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "c0f314e7-2807-4241-a792-2a6c79ed4fd0")
			)
			String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to delete {} with id {} ", Context.NAME, uuid);
		InnerMethodName.set("deleteContext");
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		contextManagement.delete();
		
		return Response.status(Status.NO_CONTENT).build();
	}
	
}
