package org.gcube.resourcemanagement.contexts.impl.entities;

import java.io.Serial;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.informationsystem.contexts.impl.entities.ContextImpl;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.model.reference.properties.Event;
import org.gcube.resourcemanagement.contexts.impl.properties.BasicInformation;

/**
 * Extension of Context specific for gCube Resource Management
 * It contains additional properties like events, information, key, availableAt
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonTypeName(value=Context.NAME)
@JsonInclude(JsonInclude.Include.NON_NULL)
public class GCubeContext extends ContextImpl {

	/**
	 * Generated Serial Version UID
	 */
	@Serial
	private static final long serialVersionUID = -8929392680534884169L;

	/**
	 * The events occurred to the Contexts.
	 * creation, renaming, parent change.
	 * Some of the event are managed by the resource-registry.
	 * Others can be added by an authorized client.
	 * This create a sort of journal. See #27707
	 */
	public static final String EVENTS_PROPERTY = "events";
	
	/**
	 * It contains the basic information for the context
	 */
	public static final String INFORMATION_PROPERTY = "information";
	
	/**
	 * This information is provided to allowed user only (by role)
	 * The symmetric key for the context
	 */
	public static final String KEY_PROPERTY = "key";
	
	/**
	 * {
	 * 	...
	 * 	"availableAt" : [
	 * 		"https://i-marine.d4science.org/group/alienandinvasivespecies", 
	 * 		"https://services.d4science.org/group/alienandinvasivespecies"
	 * 	]
	 * 	...
	 * }
	 * 
	 * For non VRE context this field could be null or could have multiple value.
	 * For VRE it is normally one value only (but some exception could exists)
	 */
	public static final String AVAILABLE_AT_PROPERTY = "availableAt";
	
	/**
	 * The events occurred to the Context
	 */
	protected SortedSet<Event> events;

	/**
	 * The basic information for the context
	 */
	protected BasicInformation information;

	/**
	 * The symmetric key for the context
	 */
	protected String key;
	
	/**
	 * The list of availableAt URLs
	 */
	protected List<String> availableAt;
	
	/**
	 * Constructs a GCubeContext from a Context
	 * 
	 * @param c the Context
	 */
	public GCubeContext(Context c) {
		this.uuid = c.getID();
		this.metadata = c.getMetadata();
		
		this.name = c.getName();
		
		this.parent = c.getParent();
		this.children = c.getChildren();
		
		this.state = c.getState();
		
		this.additionalProperties = new HashMap<>();
		Map<String, Object> ap = c.getAdditionalProperties();
		for(String key : ap.keySet()) {
			Object obj = ap.get(key);
			switch (key) {
				case EVENTS_PROPERTY:
					@SuppressWarnings("unchecked") 
					SortedSet<Event> events = (SortedSet<Event>) obj;
					setEvents(events);
					break;
	
				case KEY_PROPERTY:
					setKey((String) obj);
					break;
				
				case INFORMATION_PROPERTY:
					setInformation((BasicInformation) obj);
					break;
				
				case AVAILABLE_AT_PROPERTY:
					@SuppressWarnings("unchecked")
					List<String> availableAt = (List<String>) obj;
					setAvailableAt(availableAt);
					break;
					
				default:
					this.additionalProperties.put(key, obj);
					break;
			}
		}
		
	}
	
	/**
	 * Default constructor
	 */
	protected GCubeContext() {
		super();
	}
	
	/**
	 * Constructs a GCubeContext with the given UUID
	 * @param uuid the UUID
	 */
	public GCubeContext(UUID uuid) {
		this(null, uuid);
	}
	
	/**
	 * Constructs a GCubeContext with the given name
	 * @param name the name
	 */
	public GCubeContext(String name) {
		this(name, null);
	}
	
	/**
	 * Constructs a GCubeContext with the given name and UUID
	 * @param name the name
	 * @param uuid the UUID
	 */
	public GCubeContext(String name, UUID uuid) {
		super(name, uuid);
	}
	
	/**
	 * Sets the events occurred to the Context
	 * @return the events
	 */
	@JsonGetter(EVENTS_PROPERTY)
	public SortedSet<Event> getEvents() {
		return events;
	}

	/**
	 * Sets the events occurred to the Context
	 * @param events the events to set
	 */
	@JsonSetter(EVENTS_PROPERTY)
	public void setEvents(SortedSet<Event> events) {
		this.events = events;
	}
	
	/**
	 * Adds an event to the Context.
	 * creation, renaming, parent change.
	 * Some of the event are managed by the resource-registry.
	 * Others can be added by an authorized client.
	 * This create a sort of journal. See #27707
	 * @param event the event to add
	 */
	@JsonIgnore
	public void addEvent(Event event) {
		if(this.events==null) {
			this.events = new TreeSet<>();
		}
		this.events.add(event);
	}

	/**
	 * Gets the basic information for the context
	 * @return the information
	 */
	@JsonGetter(INFORMATION_PROPERTY)
	public BasicInformation getInformation() {
		return information;
	}

	/**
	 * Sets the basic information for the context
	 * @param information the information to set
	 */
	@JsonSetter(INFORMATION_PROPERTY)
	public void setInformation(BasicInformation information) {
		this.information = information;
	}

	/**
	 * Gets the symmetric key for the context
	 * @return the key
	 */
	@JsonGetter(KEY_PROPERTY)
	public String getKey() {
		return key;
	}

	/**
	 * Sets the symmetric key for the context
	 * @param key the key to set
	 */
	@JsonSetter(KEY_PROPERTY)
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * Gets the list of availableAt URLs
	 * @return the availableAt
	 */
	@JsonGetter(AVAILABLE_AT_PROPERTY)
	public List<String> getAvailableAt() {
		return availableAt;
	}

	/**
	 * Sets the list of availableAt URLs
	 * @param availableAt the availableAt to set
	 */
	@JsonSetter(AVAILABLE_AT_PROPERTY)
	public void setAvailableAt(List<String> availableAt) {
		this.availableAt = availableAt;
	}
	
}
