package org.gcube.common.security.providers;

import org.gcube.common.security.secrets.Secret;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides a thread-safe way to manage and access a {@link Secret} object
 * within the current execution thread.
 * <p>
 * This class uses an {@link InheritableThreadLocal} to ensure that a secret
 * set in a parent thread is automatically inherited by any child threads
 * that are created, which is crucial for managing authorization contexts
 * in asynchronous or multi-threaded environments.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class SecretManagerProvider {

	/**
	 * Logger instance for logging purposes.
	 */
	private static final Logger logger = LoggerFactory.getLogger(SecretManagerProvider.class);

	/**
	 * A thread-local variable that holds the {@link Secret} for the current thread.
	 * The use of {@code InheritableThreadLocal} means that this value is also
	 * passed down to child threads.
	 */
	private static final InheritableThreadLocal<Secret> thread = new InheritableThreadLocal<Secret>() {
		@Override
		protected Secret initialValue() {
			return null;
		}
	};
	
	/**
	 * Retrieves the {@link Secret} associated with the current thread.
	 *
	 * @return The {@link Secret} object for the current thread, or {@code null}
	 * if no secret has been set.
	 */
	public static Secret get() {
		return thread.get();
	}
	
	/**
	 * Sets the {@link Secret} for the current thread.
	 * <p>
	 * This secret will be accessible by the current thread and any child
	 * threads it creates.
	 *
	 * @param secret The {@link Secret} object to be set.
	 */
	public static void set(Secret secret) {
		Secret current = get();
		if (current == null) {
			thread.set(secret);
		}else {
			if(secret.getOwner().getId().compareTo(current.getOwner().getId())!=0) {
				String msg = String.format("You are trying to set a secret for owner %s but a secret for owner %s is already set in this thread. Use reset() if you want to clear the current secret before setting a new one. If, instead, you are trying to perform a nested operation with a different owner, please use AuthorizedTask instead.", secret.getOwner(), current.getOwner());
				logger.error(msg);
				throw new IllegalStateException(msg);

			}
			if (secret.getContext().compareTo(current.getContext())!=0) {
				String msg = String.format("You are trying to set a secret for context %s but a secret for context %s is already set in this thread. Use reset() if you want to clear the current secret before setting a new one. If, instead, you are trying to perform a nested operation in a different context, please use AuthorizedTask instead.", secret.getContext(), current.getContext());
				logger.error(msg);
				throw new IllegalStateException(msg);
			}
			if (secret.priority() > current.priority()) {
				thread.set(secret);
			}else {
				logger.warn("Ignoring secret {} with priority {} because the secret {} with higher priority {} is already set",
						secret.getClass().getSimpleName(), secret.priority(), current.getClass().getSimpleName(), current.priority());
			}
		}
	}
	
	/**
	 * Resets the {@link Secret} for the current thread, effectively removing it.
	 * <p>
	 * This method should be called to clean up the thread-local secret,
	 * especially after a task is completed, to prevent security issues.
	 */
	public static void reset() {
		thread.remove();
	}
}