package org.gcube.common.security;



/**
 * A data object model representing a hierarchical scope.
 * <p>
 * This class provides a structured representation for scopes that are
 * defined in a linear, path-like syntax (e.g., "/INFRASTRUCTURE/VO/VRE").
 * It supports three types of scopes: {@link Type#INFRASTRUCTURE}, {@link Type#VO},
 * and {@link Type#VRE}. The class allows for navigating the scope hierarchy
 * and provides utility methods for comparison and string representation.
 *
 * @author Fabio Simeoni
 */
public class ContextBean {

	/**
	 * Separator used in the linear syntax of scopes.
	 */
	protected static String separator = "/";

	/**
	 * Enumeration for the different types of scopes supported.
	 * The hierarchy is INFRASTRUCTURE -> VO -> VRE.
	 */
	public static enum Type implements Comparable<Type> {
		/**
		 * Represents the top-level infrastructure scope.
		 */
		INFRASTRUCTURE,
		/**
		 * Represents a Virtual Organization scope, enclosed within an infrastructure.
		 */
		VO,
		/**
		 * Represents a Virtual Research Environment scope, enclosed within a VO.
		 */
		VRE
	}
		
	/**
	 * The name of the scope at this level.
	 */
	private String name;
	
	/**
	 * The type of the scope (e.g., INFRASTRUCTURE, VO, VRE).
	 */
	private Type type;
	
	/**
	 * The enclosing scope in the hierarchy, if any. This is {@code null} for
	 * top-level scopes (INFRASTRUCTURE).
	 */
	private ContextBean enclosingScope;
	
	
	/**
	 * Returns the name of this scope level.
	 *
	 * @return The name of the scope.
	 */
	public String name() {
		return name;
	}

	/**
	 * Checks if the scope has a specific {@link Type}.
	 *
	 * @param type The {@link Type} to check against.
	 * @return {@code true} if the scope's type matches the given type, otherwise {@code false}.
	 */
	public boolean is(Type type) {
		return this.type.equals(type);
	}
	
	/**
	 * Returns the {@link Type} of this scope.
	 *
	 * @return The type of the scope.
	 */
	public Type type() {
		return type;
	}
	
	/**
	 * Returns the enclosing scope in the hierarchy.
	 *
	 * @return The parent {@link ContextBean} object, or {@code null} if this is
	 * a top-level scope (type is INFRASTRUCTURE).
	 */
	public ContextBean enclosingScope() {
		return enclosingScope;
	}
	
	/**
	 * Constructs a new {@code ContextBean} from a linear string representation.
	 * The constructor parses the string to determine the scope's name, type,
	 * and its enclosing scope.
	 *
	 * @param scope The linear string representing the scope (e.g., "/INFRASTRUCTURE/VO/VRE").
	 * @throws IllegalArgumentException if the provided scope string is malformed
	 * or does not conform to the expected hierarchy.
	 */
	public ContextBean(String scope) throws IllegalArgumentException {
		
		String[] components = scope.split(separator);
		
		if (components.length < 2 || components.length > 4) {
			throw new IllegalArgumentException("Scope '" + scope + "' is malformed. Expected format: /INFRASTRUCTURE or /INFRASTRUCTURE/VO or /INFRASTRUCTURE/VO/VRE.");
		}
		
		if (components.length > 3) {
			this.name = components[3];
			this.enclosingScope = new ContextBean(separator + components[1] + separator + components[2]);
			this.type = Type.VRE;
		} else if (components.length > 2) {
			this.name = components[2];
			this.enclosingScope = new ContextBean(separator + components[1]);
			this.type = Type.VO;
		} else {
			this.name = components[1];
			this.type = Type.INFRASTRUCTURE;
		}
	}
	
	/**
	 * Returns the linear string representation of the scope.
	 * This method reconstructs the path from the current scope to the top-level.
	 *
	 * @return The linear string representation (e.g., "/INFRASTRUCTURE/VO").
	 */
	@Override
	public String toString() {
		return is(Type.INFRASTRUCTURE) ? separator + name() : enclosingScope().toString() + separator + name();
	}

	/**
	 * Calculates the hash code for this object based on its name, type, and enclosing scope.
	 *
	 * @return A hash code value for this object.
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((enclosingScope == null) ? 0 : enclosingScope.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	/**
	 * Compares this object with another object for equality.
	 *
	 * @param obj The object to compare with.
	 * @return {@code true} if the objects are the same or have the same name, type,
	 * and enclosing scope; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ContextBean other = (ContextBean) obj;
		if (enclosingScope == null) {
			if (other.enclosingScope != null)
				return false;
		} else if (!enclosingScope.equals(other.enclosingScope))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (type != other.type)
			return false;
		return true;
	}
}