/*
 * Decompiled with CFR 0.152.
 */
package org.gcube.dataanalysis.oscar;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Calendar;
import java.util.List;
import java.util.Vector;
import org.apache.commons.io.FileSystemUtils;
import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.oscar.OscarMerger;
import org.gcube.dataanalysis.oscar.util.FTPDownloader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.Attribute;
import ucar.nc2.FileWriter2;
import ucar.nc2.NetcdfFile;
import ucar.nc2.NetcdfFileWriter;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.util.CancelTask;
import ucar.nc2.util.CancelTaskImpl;

public class IncrementalOscarMerger {
    private static final Integer OSCAR_START_YEAR = 1992;
    private String yearlyOscarRemoteFilePattern = "ftp://podaac-ftp.jpl.nasa.gov/allData/oscar/preview/L4/resource/LAS/oscar_third_deg_180/oscar_vel${YYYY}_180.nc";
    private String yearlyOscarLocalFilePattern = "oscar_vel${YYYY}_180.nc";
    private static String workdir = "/tmp/oscar-merger";
    private static String mergedFilePattern = "oscar_vel_${FROMYEAR}-${TOYEAR}_180.nc";
    private static String mergeDescriptorFilePattern = "oscar_vel_${FROMYEAR}-${TOYEAR}_180.xml";
    private Integer computedStartYear;
    private Integer computedEndYear;
    private Integer intervalSize = 3;
    private Boolean removeTmpFiles = true;
    private Boolean debug = false;
    private static final Logger logger = LoggerFactory.getLogger(OscarMerger.class);

    private Integer getStartYear() {
        if (this.computedStartYear == null) {
            int year;
            int n = year = this.debug != false ? Calendar.getInstance().get(1) - 1 : OSCAR_START_YEAR;
            while (year <= Calendar.getInstance().get(1)) {
                if (this.checkRemoteYearFileExist(year)) {
                    this.computedStartYear = year;
                    logger.info("Setting start year of the merge is " + this.computedStartYear);
                    break;
                }
                ++year;
            }
        }
        return this.computedStartYear;
    }

    private Integer getEndYear() {
        if (this.computedEndYear == null) {
            for (int year = Calendar.getInstance().get(1); year > OSCAR_START_YEAR; --year) {
                if (!this.checkRemoteYearFileExist(year)) continue;
                this.computedEndYear = year;
                logger.info("Setting end year of the merge is " + this.computedEndYear);
                break;
            }
        }
        return this.computedEndYear;
    }

    private File getLastUsefulMergedFile() {
        int endYear;
        int startYear = this.getStartYear();
        for (int i = endYear = this.getEndYear().intValue(); i >= startYear; --i) {
            File f = new File(this.getWorkDir(), this.formatMergedFileName(startYear, i));
            if (!f.exists() || this.canStillChange(i)) continue;
            return f;
        }
        return null;
    }

    private boolean canStillChange(int year) {
        int currentYear = Calendar.getInstance().get(1);
        if (year == currentYear) {
            return true;
        }
        return year == currentYear - 1 && !this.checkRemoteYearFileExist(currentYear);
    }

    private boolean checkRemoteYearFileExist(int year) {
        return FTPDownloader.checkFtpFileExists(this.getSourceURLForYear(year));
    }

    private String getSourceURLForYear(Integer year) {
        return this.yearlyOscarRemoteFilePattern.replaceAll("\\$\\{YYYY\\}", year + "");
    }

    private File getLocalFileForYear(Integer year) {
        return new File(this.getWorkDir(), this.yearlyOscarLocalFilePattern.replaceAll("\\$\\{YYYY\\}", year + ""));
    }

    private File getMergedFilesForYears(Integer from, Integer to) {
        return new File(this.getWorkDir(), this.formatMergedFileName(from, to));
    }

    private File generateDescriptor(List<String> paths, File descriptorFile) {
        String out = "";
        out = out + "<netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\">\n";
        out = out + "  <attribute name=\"title\" value=\"OSCAR Velocity Dataset\"/>\n";
        out = out + "  <aggregation type=\"joinExisting\" dimName=\"time\">\n";
        for (String p : paths) {
            out = out + "    <netcdf location=\"" + p + "\"/>\n";
        }
        out = out + "  </aggregation>\n";
        out = out + "</netcdf>";
        try (PrintWriter pw = new PrintWriter(descriptorFile);){
            pw.println(out);
        }
        catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return descriptorFile;
    }

    private File getWorkDir() {
        File out = new File(workdir);
        out.mkdir();
        return out;
    }

    private String formatMergedFileName(Integer fromYear, Integer toYear) {
        return mergedFilePattern.replaceAll("\\$\\{FROMYEAR\\}", fromYear.toString()).replaceAll("\\$\\{TOYEAR\\}", toYear.toString());
    }

    private String formatDescriptorFileName(Integer fromYear, Integer toYear) {
        return mergeDescriptorFilePattern.replaceAll("\\$\\{FROMYEAR\\}", fromYear.toString()).replaceAll("\\$\\{TOYEAR\\}", toYear.toString());
    }

    private Integer extractEndYear(String fileName) {
        for (int year = this.getStartYear().intValue(); year <= this.getEndYear(); ++year) {
            String name = this.formatMergedFileName(this.getStartYear(), year);
            if (fileName.equals(name)) {
                return year;
            }
            name = this.formatDescriptorFileName(this.getStartYear(), year);
            if (!fileName.equals(name)) continue;
            return year;
        }
        return null;
    }

    private int getIntervalSize() {
        return this.intervalSize;
    }

    public String merge() throws Exception {
        logger.info("Doing some preliminary cleanup...");
        this.cleanup();
        this.checkReady();
        File lastMerged = this.getLastUsefulMergedFile();
        if (lastMerged != null) {
            logger.info("Found a previously merged file: " + lastMerged.getAbsolutePath());
        } else {
            File currentYearOscarFile = this.getLocalFileForYear(this.getStartYear());
            logger.info("Downloading file to " + currentYearOscarFile.getAbsolutePath());
            this.downloadFile(this.yearlyOscarRemoteFilePattern.replaceAll("\\$\\{YYYY\\}", this.getStartYear() + ""), currentYearOscarFile, false);
            lastMerged = this.getMergedFilesForYears(this.getStartYear(), this.getStartYear());
            logger.info("Renaming it as " + lastMerged.toString());
            FileUtils.moveFile((File)FileUtils.getFile((File)currentYearOscarFile, (String[])new String[0]), (File)FileUtils.getFile((File)lastMerged, (String[])new String[0]));
        }
        logger.info("End year of the last merged file is " + this.extractEndYear(lastMerged.getName()));
        int year = this.extractEndYear(lastMerged.getName()) + 1;
        while (year <= this.getEndYear()) {
            Vector<String> paths = new Vector<String>();
            paths.add(lastMerged.getAbsolutePath());
            int intervalEnd = year;
            for (int i = 0; i < this.getIntervalSize() && year + i <= this.getEndYear(); ++i) {
                intervalEnd = year + i;
                File currentYearOscarFile = this.getLocalFileForYear(intervalEnd);
                logger.info("Downloading file for year " + intervalEnd);
                this.downloadFile(this.getSourceURLForYear(intervalEnd), currentYearOscarFile, false);
                paths.add(currentYearOscarFile.getAbsolutePath());
            }
            logger.info("Generating descriptor...");
            File mergeDescriptor = new File(this.getWorkDir(), this.formatDescriptorFileName(this.getStartYear(), intervalEnd));
            this.generateDescriptor(paths, mergeDescriptor);
            logger.info("Descriptor is " + mergeDescriptor.getAbsolutePath());
            File mergedFile = this.getMergedFilesForYears(this.getStartYear(), intervalEnd);
            logger.info("Merging files at " + mergedFile.getAbsolutePath());
            this.mergeDescribedFilesTo(mergeDescriptor, mergedFile);
            logger.info("Merged");
            if (!this.canStillChange(this.extractEndYear(mergedFile.getName()))) {
                logger.info("Removing " + lastMerged.getName() + " since the merged is stable. We'll keep it instead.");
                this.removeFileOrDir(lastMerged);
            } else {
                logger.info("Keeping " + lastMerged.getName() + " since the merged one is not yet stable.");
            }
            logger.info("Removing descriptor: " + mergeDescriptor.getAbsolutePath());
            this.removeFileOrDir(mergeDescriptor);
            logger.info("Remove yearly files... ");
            for (int i = year; i <= intervalEnd; ++i) {
                logger.info("Removing file " + this.getLocalFileForYear(i).getName());
                this.removeFileOrDir(this.getLocalFileForYear(i));
            }
            lastMerged = mergedFile;
            year = intervalEnd + 1;
        }
        return lastMerged.getAbsolutePath();
    }

    private void downloadFile(String source, File destination, Boolean forceDownload) throws IOException, Exception {
        if (forceDownload == null) {
            forceDownload = false;
        }
        if (!forceDownload.booleanValue() && destination.exists()) {
            logger.info("File already downloaded and completed, skipping.");
            return;
        }
        if (!source.startsWith("ftp:")) {
            throw new Exception("Only ftp downloads currently supported");
        }
        String tmpFileName = destination.getAbsolutePath() + ".tmp";
        FTPDownloader ftpDownloader = new FTPDownloader(source);
        ftpDownloader.downloadFile(source, tmpFileName);
        logger.info("FTP File downloaded successfully");
        ftpDownloader.disconnect();
        FileUtils.moveFile((File)FileUtils.getFile((String[])new String[]{tmpFileName}), (File)FileUtils.getFile((String[])new String[]{destination.getAbsolutePath()}));
    }

    private void mergeDescribedFilesTo(File descriptor, File merged) throws Exception {
        String datasetIn = descriptor.getAbsolutePath();
        String datasetOut = merged.getAbsolutePath() + ".tmp";
        CancelTaskImpl cancel = new CancelTaskImpl();
        NetcdfFile ncfileIn = NetcdfDataset.openFile((String)datasetIn, (CancelTask)cancel);
        logger.info(String.format("NetcdfDatataset read from %s write to %s ", datasetIn, datasetOut));
        NetcdfFileWriter.Version version = NetcdfFileWriter.Version.netcdf4;
        FileWriter2 writer = new FileWriter2(ncfileIn, datasetOut, version, null);
        writer.getNetcdfFileWriter().setLargeFile(true);
        NetcdfFile ncfileOut = writer.write((CancelTask)cancel);
        for (Attribute a : ncfileOut.getGlobalAttributes()) {
            logger.info(a.toString());
        }
        if (ncfileOut != null) {
            ncfileOut.close();
        }
        ncfileIn.close();
        cancel.setDone(true);
        FileUtils.moveFile((File)FileUtils.getFile((String[])new String[]{datasetOut}), (File)merged);
        logger.info(String.format("%s%n", cancel));
    }

    public void cleanup() {
        this.cleanup(false);
    }

    public void cleanup(boolean complete) {
        if (complete) {
            this.removeFileOrDir(this.getWorkDir());
        } else {
            Vector<String> filesToKeep = new Vector<String>();
            File lastUseful = this.getLastUsefulMergedFile();
            if (lastUseful != null) {
                filesToKeep.add(lastUseful.getAbsolutePath());
            }
            for (File f : this.getWorkDir().listFiles()) {
                if (filesToKeep.contains(f.getAbsolutePath())) {
                    logger.info("Keeping " + f.getAbsolutePath());
                    continue;
                }
                this.removeFileOrDir(f);
            }
        }
    }

    private void removeFileOrDir(File file) {
        logger.info("Removing " + file.getAbsolutePath());
        if (!file.getAbsolutePath().startsWith("/tmp")) {
            logger.info("NOT REMOVING ANYTHING OUTSIDE /tmp");
            return;
        }
        if (file.getAbsolutePath().endsWith(".keep")) {
            logger.info("NOT REMOVING " + file.getAbsolutePath() + " since it has to be kept");
            return;
        }
        if (this.removeTmpFiles.booleanValue()) {
            FileUtils.deleteQuietly((File)file);
            logger.info("Removed");
        } else {
            logger.info("Removing IS DISABLED");
        }
    }

    public void checkReady() throws Exception {
        this.checkEnoughDiskSpace();
    }

    private void checkEnoughDiskSpace() throws Exception {
        logger.info("Checking needed space...");
        Long backMergedFileSize = 0L;
        File f = this.getLastUsefulMergedFile();
        if (f == null) {
            backMergedFileSize = 417518585L * (long)((double)(this.getEndYear() - this.intervalSize - this.getStartYear()) - 1.2);
        }
        logger.info("Size of the last useful merge: " + backMergedFileSize);
        Long newMergedFileSize = 417518585L * (long)((double)(this.getEndYear() - this.getStartYear()) - 1.2);
        logger.info("Size of the new merge: " + newMergedFileSize);
        Long yearlyFilesSize = 1198005516L * (long)this.intervalSize.intValue();
        logger.info("Yearly files: " + yearlyFilesSize);
        Long neededSpace = backMergedFileSize + newMergedFileSize + yearlyFilesSize;
        logger.info("Needed disk space: " + neededSpace + " bytes");
        Long availableSpace = FileSystemUtils.freeSpaceKb((String)this.getWorkDir().getAbsolutePath()) * 1024L;
        logger.info("Available disk space: " + availableSpace + " bytes");
        if (neededSpace >= availableSpace) {
            Double gb = (double)neededSpace.longValue() / 1024.0 / 1024.0 / 1024.0;
            String message = String.format("Not enough disk space. At least %2.2fGB of available disk needed.", gb);
            throw new Exception(message);
        }
        logger.info("There's enough disk space to proceed");
        Double percent = (double)neededSpace.longValue() * 1.0 / (double)availableSpace.longValue() * 1.0 * 100.0;
        String message = String.format("I'm about to use %.0f%% of the available disk space (but I'll release it afterwards, I promise)", percent);
        logger.info(message);
    }

    public void setIntervalSize(Integer intervalSize) {
        if (intervalSize != null) {
            this.intervalSize = Math.max(intervalSize, 1);
        }
    }

    public void setRemoveTmpFiles(Boolean removeTmpFiles) {
        this.removeTmpFiles = removeTmpFiles;
    }

    public void setDebug(Boolean debug) {
        this.debug = debug;
        this.computedStartYear = null;
    }
}

