package com.finconsgroup.itserr.marketplace.discussion.bs.messaging.producer.impl;


import com.finconsgroup.itserr.marketplace.discussion.bs.config.properties.DiscussionBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.producer.DiscussionEventProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.discussion.bs.utils.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.discussion.bs.utils.Constants.SERVICE_FILTER_VALUE;


/**
 * Default implementation of {@link DiscussionEventProducer} that publishes
 * resource events (create, update, delete) for discussion events.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultDiscussionEventProducer implements DiscussionEventProducer {

    private final MessageProducer messageProducer;

    private final DiscussionBsConfigurationProperties.MessagingProperties messagingProperties;


    /**
     * Constructs the DefaultEventProducer.
     *
     * @param discussionBsConfigurationProperties application-specific configuration properties
     * @param producerRegistry                    the registry providing message producers
     */
    public DefaultDiscussionEventProducer(
            DiscussionBsConfigurationProperties discussionBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = discussionBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getEventProducerName());
    }

    @Override
    public void publishCreateEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getResourceThreadCreatedEventType());
    }


    /**
     * Sends a message to the messaging system.
     *
     * @param eventDto  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull MessagingEventDto<?> eventDto, @NonNull String eventType) {
        try {
            Map<String, MessagingEventDto<?>> messageMap = Map.of(eventDto.getId(), eventDto);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: %s, eventType: %s, source: %s, filterProperties: %s"
                    .formatted(messageMap, eventType, messagingProperties.getSource(), filterProperties));
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for discussion with the following info: {}", eventDto);
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage, e);
        }
    }

}
