package com.finconsgroup.itserr.marketplace.event.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.event.dm.api.EventApi;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.service.EventService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;
import java.util.UUID;

/**
 * REST controller class for handling user event related API requests.
 *
 * <p>Implements the {@link EventApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class EventController implements EventApi {

    private final EventService eventService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @Override
    public Page<OutputEventDto> findAll(
            final Set<String> associationsToLoad,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Sort.Direction direction
    ) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return eventService.findAll(associationsToLoad, pageable);
    }

    @Override
    public OutputEventDto findById(final UUID eventId) {
        return eventService.findById(JwtTokenHolder.getUserId().orElse(null), eventId);
    }

    @Override
    public OutputEventDto createEvent(final InputCreateEventDto inputCreateEventDto) {
        return eventService.create(JwtTokenHolder.getUserIdOrThrow(), inputCreateEventDto);
    }

    @Override
    public OutputEventDto updateEventById(
            final UUID eventId,
            final InputUpdateEventDto inputUpdateEventDto
    ) {
        return eventService.updateById(JwtTokenHolder.getUserIdOrThrow(), eventId, inputUpdateEventDto);
    }

    @Override
    public void deleteEventById(final UUID eventId) {
        eventService.deleteById(JwtTokenHolder.getUserIdOrThrow(), eventId);
    }

    @Override
    public OutputEventDto register(UUID id) {
        return eventService.register(JwtTokenHolder.getUserIdOrThrow(), id);
    }

    @Override
    public OutputEventDto unregister(UUID id) {
        return eventService.unregister(JwtTokenHolder.getUserIdOrThrow(), id);
    }
}
