package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.converter.StringListConverter;
import com.finconsgroup.itserr.marketplace.event.dm.model.InstitutionalPageModel;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.Lob;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.OrderColumn;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.ENUM_FIELD_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.LINK_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.LONG_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;

/**
 * Entity class representing a event in the event system.
 * This entity is mapped to the "event" table in the database and stores
 * information such as the ID, title, event type, content.
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "event")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class EventEntity extends AbstractUUIDEntity {

    /**
     * The title of the event.
     */
    @Column(name = "title", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String title;

    /**
     * The ID of the user who planned the event.
     */
    @Column(name = "event_planner_id", nullable = false)
    private UUID eventPlannerId;

    /**
     * The link of the event.
     */
    @Column(name = "event_link", length = LINK_TEXT_LENGTH)
    private String eventLink;

    /**
     * The type of event.
     */
    @Column(name = "event_type", nullable = false, length = ENUM_FIELD_LENGTH)
    private String eventType;

    /**
     * The content of the event.
     */
    @Column(name = "content", length = LONG_TEXT_LENGTH)
    private String content;

    /**
     * The number of participants subscribed for the event.
     */
    @Column(name = "subscribed_participants")
    private Integer subscribedParticipantsCount;

    /**
     * List of tags associated with the event.
     */
    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "tags")
    private List<String> tags;

    /**
     * JSON structure for list of institutional pages.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "institutional_pages", columnDefinition = "json")
    private List<InstitutionalPageModel> institutionalPages;

    /**
     * The document ID of the image.
     */
    @Column(name = "image_id")
    private UUID imageId;

    /**
     * Public URL of the image.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * Timezone of the event.
     */
    @Column(name = "timezone", nullable = false)
    private String timezone;

    /**
     * Start Date of the event.
     */
    @Column(name = "start_date", nullable = false)
    private LocalDate startDate;

    /**
     * End Date of the event.
     */
    @Column(name = "end_date", nullable = false)
    private LocalDate endDate;

    /**
     * The ID of the user who created the event.
     */
    @Column(name = "maintainer_id", nullable = false)
    private UUID maintainerId;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The event conductors related to the event.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "event")
    @OrderColumn(name = "conductor_order")
    @ToString.Exclude
    private List<EventConductorEntity> eventConductors;

    /**
     * The schedules related to the event.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "event")
    @OrderColumn(name = "schedule_order")
    @ToString.Exclude
    private List<ScheduleEntity> schedules;

    /**
     * The subscribed participants related to the event.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.REMOVE, orphanRemoval = true, mappedBy = "event")
    @OrderBy("creationTime ASC")
    @ToString.Exclude
    private List<SubscribedParticipantEntity> subscribedParticipants;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }


    public void incrementParticipants() {
        this.subscribedParticipantsCount++;
    }

    public void decrementParticipants() {
        if (this.subscribedParticipantsCount == 0) {
            return;
        }
        this.subscribedParticipantsCount--;
    }
}
