package com.finconsgroup.itserr.marketplace.institutional_page.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputPendingMemberRequestsDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to member management for InstitutionalPages.
 *
 * <p>
 * It provides endpoints for join requests, invitations, accept/reject of join-requests/invitations, and promotion to WP Leader.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET    /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests                - Retrieves all pending member join requests and invitations for a specific InstitutionalPage (WP Leader)
 * GET    /api/v1/bs/institutional-page/institutional-pages/members/requests                                      - Retrieves all pending member join requests and invitations related to the authenticated user
 * POST   /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations    - Invites users to join the specified InstitutionalPage as members (WP Leader)
 * DELETE /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations/{invitedUserId}   - Cancels an invitation for a specific user to join an InstitutionalPage (WP Leader)
 * PATCH  /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations    - Accepts/Rejects an invitation to join an InstitutionalPage
 * PATCH  /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/membership              - Updates membership of members to WP Leader/Member for an InstitutionalPage (WP Leader)
 * POST   /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join           - Submits a join request for the authenticated user to join an InstitutionalPage
 * DELETE /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join           - Cancels a join request previously submitted by the authenticated user
 * PATCH  /api/v1/bs/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join           - Accepts/Rejects join requests to an institutional page (WP Leader)
 * GET    /api/v1/dm/institutional-page/institutional-pages/{rootInstitutionalPageId}/members/hierarchy           - Retrieves paginated list of all users belonging to the hierarchy of a given root InstitutionalPage (WP Leader)
 * </pre>
 * </p>
 */
@Tag(
        name = "Member",
        description = "The Member API: It provides endpoints for " +
                "join requests, invitations, accept/reject of join-requests/invitations, and promotion to WP Leader"
)
@SecurityRequirement(name = "BearerAuth")
public interface MemberApi {

    /**
     * Retrieves all pending member join requests and invitations for a specific InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves all pending member join requests and invitations for a specific InstitutionalPage (WP Leader)",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto findPendingMemberRequestsForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Retrieves all pending member join requests and invitations related to the authenticated user.
     *
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves all pending member join requests and invitations related to the authenticated user",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/members/requests",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto findPendingMemberRequestsForUser();

    /**
     * Invites users to join the specified InstitutionalPage as members (WP Leader).
     *
     * @param institutionalPageId   the id of the InstitutionalPage
     * @param inputInviteMembersDto the DTO for inviting users to join the InstitutionalPage as members
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Invites users to join the specified InstitutionalPage as members (WP Leader)",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputPendingMemberRequestsDto inviteMembers(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputInviteMembersDto inputInviteMembersDto
    );

    /**
     * Cancels an invitation for a specific user to join an InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @param invitedUserId       the id of the invited user
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Cancels an invitation for a specific user to join an InstitutionalPage (WP Leader)",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations/{invitedUserId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto cancelInvitation(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("invitedUserId") UUID invitedUserId
    );

    /**
     * Accepts/Rejects an invitation to join an InstitutionalPage
     *
     * @param institutionalPageId              the id of the InstitutionalPage
     * @param InputPatchIPInvitationRequestDto the DTO to approve or reject invitation to an Institutional Page
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Accepts/Rejects an invitation to join an InstitutionalPage",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto acceptOrRejectInvitation(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputPatchIPInvitationRequestDto InputPatchIPInvitationRequestDto
    );

    /**
     * Updates membership of members to WP Leader/Member for an InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId     the id of the InstitutionalPage
     * @param inputPatchMembershipDto Input DTO for updating membership of users
     * @return {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Updates membership of members to WP Leader/Member for an InstitutionalPage (WP Leader)",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/membership",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto patchMembershipOfUsersForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputPatchMembershipDto inputPatchMembershipDto
    );

    /**
     * Remove membership of users for an InstitutionalPage.
     *
     * @param institutionalPageId      the id of the InstitutionalPage
     * @param inputRemoveMembershipDto Input DTO for removing membership of users
     * @return {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Remove membership of users for an InstitutionalPage",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/membership",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto removeMembershipOfUsersForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputRemoveMembershipDto inputRemoveMembershipDto
    );


    /**
     * Submits a join request for the authenticated user to join an InstitutionalPage.
     *
     * @param institutionalPageId       the id of the InstitutionalPage
     * @param inputSubmitJoinRequestDto the DTO Input DTO for submitting a join request
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Submits a join request for the authenticated user to join an InstitutionalPage",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputPendingMemberRequestsDto submitJoinRequest(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputSubmitJoinRequestDto inputSubmitJoinRequestDto
    );

    /**
     * Cancels a join request previously submitted by the authenticated user.
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Cancels a join request previously submitted by the authenticated user",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto cancelJoinRequest(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Accepts/Rejects join requests to an institutional page (WP Leader).
     *
     * @param institutionalPageId        the id of the InstitutionalPage
     * @param inputPatchIPJoinRequestDto the DTO to approve or reject a pending join request to an Institutional Page
     * @return {@link OutputPendingMemberRequestsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Accepts/Rejects join requests to an institutional page (WP Leader)",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPendingMemberRequestsDto acceptOrRejectJoinRequests(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    );

    /**
     * Retrieves paginated list of all users belonging
     * to the hierarchy of a given root InstitutionalPage (WP Leader).
     *
     * @param rootInstitutionalPageId ID of the root institutional page
     * @param pageNumber              the page number to retrieve (default is 0)
     * @param pageSize                the number of InstitutionalPages per page (default is 10)
     * @return a page of {@link OutputMembersInHierarchyDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves paginated list of all users belonging" +
                    " to the hierarchy of a given root InstitutionalPage (WP Leader).",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/{rootInstitutionalPageId}/members/hierarchy",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPageDto<OutputMembersInHierarchyDto> findAllUsersInHierarchy(
            @PathVariable("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize
    );

}
