package com.finconsgroup.itserr.marketplace.metadata.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputInternalFindMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.ArchivedMetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataFieldEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.dm.exception.MetadataExistsException;
import com.finconsgroup.itserr.marketplace.metadata.dm.exception.MetadataNotFoundException;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.ArchivedMetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.MetadataFieldMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.ArchivedMetadataRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.MetadataFieldRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.MetadataRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification.MetadataSpecifications;
import com.finconsgroup.itserr.marketplace.metadata.dm.service.MetadataService;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;

/**
 * Default implementation of {@link MetadataService} to perform operations related
 * to metadata resources
 */
@Service
@RequiredArgsConstructor
public class DefaultMetadataService implements MetadataService {

    /**
     * Metadata repository.
     */
    private final MetadataRepository metadataRepository;

    /**
     * MetadataFieldRepository repository.
     */
    private final MetadataFieldRepository metadataFieldRepository;

    /**
     * ArchivedMetadata repository.
     */
    private final ArchivedMetadataRepository archivedMetadataRepository;

    /**
     * Metadata mapper.
     */
    private final MetadataMapper metadataMapper;

    /**
     * MetadataFieldMapper mapper.
     */
    private final MetadataFieldMapper metadataFieldMapper;

    /**
     * ArchivedMetadata mapper.
     */
    private final ArchivedMetadataMapper archivedMetadataMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetadataPreviewDto> findAll(MetadataCategoryEnum metadataCategoryEnum, @NonNull Pageable pageable) {
        if (metadataCategoryEnum != null) {
            return metadataRepository.findAllByCategory(metadataCategoryEnum, pageable)
                    .map(metadataMapper::metadataEntityToMetadataPreviewDto);
        } else {
            return metadataRepository.findAll(pageable)
                    .map(metadataMapper::metadataEntityToMetadataPreviewDto);
        }
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetadataFieldDto> findAllFieldsById(UUID metadataId, @NonNull Pageable pageable) {
        return metadataFieldRepository.findAllByMetadataId(metadataId, pageable)
                .map(metadataFieldMapper::metadataFieldEntityToMetadataFieldDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category, @NonNull Pageable pageable) {        
        Page<MetadataFieldEntity> metadataCategories;
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        if (category != null) {
            // If PERSONAL category is requested, filter only for the user's own metadata
            if (category.equals(MetadataCategoryEnum.PERSONAL)) {
                metadataCategories = metadataFieldRepository.findByCategoryInAndCreatorId( List.of(category), userId, pageable);
            } else {
                // For COMMUNITY or STANDARD, return all metadata of that category, no matter the creator
                metadataCategories = metadataFieldRepository.findByMetadataCategory(category, pageable);
            }
        } else {
            // If no category is specified, return PERSONAL metadata of the user plus all COMMUNITY and STANDARD metadata
            metadataCategories = metadataFieldRepository.findByCategoryInOrCreatorId(List.of(MetadataCategoryEnum.COMMUNITY, MetadataCategoryEnum.STANDARD), userId, pageable);
        }
        return metadataCategories
                .map(metadataFieldMapper::metadataFieldEntityToMetadataFieldExtDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputMetadataDto findById(@NonNull final UUID metadataId) {
        return metadataRepository.findById(metadataId)
                .map(metadataMapper::metadataEntityToMetadataDto)
                .orElseThrow(() -> new MetadataNotFoundException(metadataId));
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputMetadataDto create(@NonNull final InputCreateMetadataDto request, @NonNull UUID userId) {

        // Initialize
        final String name = StringUtils.trim(request.getName());

        // Checks that there is no other metadata with the same name
        if (metadataRepository.countByNameIgnoreCase(name) > 0) {
            throw new MetadataExistsException(name);
        }

        // Map to entity
        final MetadataEntity metadata = metadataMapper.metadataSaveRequestDtoToMetadataEntity(request, userId);

        // Save
        MetadataEntity savedMetadata = metadataRepository.saveAndFlush(metadata);

        // Return DTO of the saved entity
        return metadataMapper.metadataEntityToMetadataDto(savedMetadata);

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, noRollbackFor = Exception.class, readOnly = true)
    public List<OutputMetadataDto> find(@Nullable final InputInternalFindMetadataDto request) {

        // Find metadata according to criteria
        final List<MetadataEntity> foundMetadata =
                request == null
                        ? metadataRepository.findAll()
                        : metadataRepository.findAll(
                        MetadataSpecifications
                                .ids(request.getId()));

        // Return DTOs of the found entities
        return foundMetadata.stream()
                .map(metadataMapper::metadataEntityToMetadataDto)
                .toList();

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputMetadataDto deleteById(@NonNull final UUID metadataId) {
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new MetadataNotFoundException(metadataId));
        ArchivedMetadataEntity archivedMetadataEntity = archivedMetadataMapper.metadataEntityToArchivedMetadataEntity(metadataEntity);
        archivedMetadataRepository.save(archivedMetadataEntity);
        metadataRepository.delete(metadataEntity);
        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }

}
