package com.finconsgroup.itserr.marketplace.metrics.dm.service.impl;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourceMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricInterestWeightEntity;
import com.finconsgroup.itserr.marketplace.metrics.dm.mapper.MetricTypeMapper;
import com.finconsgroup.itserr.marketplace.metrics.dm.repository.MetricInterestWeightRepository;
import com.finconsgroup.itserr.marketplace.metrics.dm.service.MetricInterestService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link MetricInterestService} to perform operations related to user-related metrics interest.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetricInterestService implements MetricInterestService {

    /** Interest weights timeout in seconds. */
    private static final long INTEREST_WEIGHTS_TIMEOUT_SECS = 60 * 5;

    /** Repository instance for accessing and managing metric interest weights data */
    private final MetricInterestWeightRepository metricInterestWeightRepository;

    /** Mapper for converting between metric type DTOs and entities */
    private final MetricTypeMapper metricTypeMapper;

    /** Cached interest weights. */
    private final Map<MetricDtoType, Double> interestWeights = new HashMap<>();
    /** Timestamp of interest weights timeout. */
    private long interestWeightTimeout;

    /**
     * Loads and updates the cached interest weights if required, based on the timeout condition.
     */
    private void loadMetricsIfRequired() {

        synchronized (interestWeights) {
            if (interestWeightTimeout < System.currentTimeMillis()) {
                final Map<MetricDtoType, Double> actualInterestWeights = metricInterestWeightRepository.findAll()
                        .stream()
                        .collect(Collectors.toMap(
                                e -> metricTypeMapper.metricTypeToMetricDtoType(e.getMetric()),
                                MetricInterestWeightEntity::getWeight));
                interestWeights.clear();
                interestWeights.putAll(actualInterestWeights);
                interestWeightTimeout = System.currentTimeMillis() + (INTEREST_WEIGHTS_TIMEOUT_SECS * 1000);
            }
        }

    }

    @Override
    public double getTotalInterest(final OutputResourceMetricsDto resourceMetrics) {
        loadMetricsIfRequired();
        return resourceMetrics != null
                ? resourceMetrics.getMetrics().entrySet()
                .stream()
                .filter(e -> e.getValue() != null)
                .mapToDouble(e -> e.getValue() * interestWeights.getOrDefault(e.getKey(), 0.0))
                .sum()
                : 0;
    }

}
